<!DOCTYPE style-sheet PUBLIC "-//James Clark//DTD DSSSL Style Sheet//EN">

<style-sheet>
<style-specification>
<style-specification-body>

(define debug (external-procedure "UNREGISTERED::James Clark//Procedure::debug"))
(define language (external-procedure "UNREGISTERED::OpenJade//Procedure::language"))

; !!! OpenJade / JadeTeX extension; will not work with other DSSSL engines !!!
(declare-characteristic page-n-columns "UNREGISTERED::James Clark//Characteristic::page-n-columns" 1)


; ================
; GLOBAL VARIABLES
; ================

; These variables should be superseded via a -V option on openjade command line
(define *document-title* "")
(define *product-version* "")

; This variable may be modified here or via a -V option on openjade command line
(define *product-name* "My Software Product")


; The following markers will be replaced by RTDS when exporting the SGML document

;%HEADING_STYLES%

;%PARAGRAPH_STYLES%

;%CHARACTER_STYLES%


; Global variables for page setup
(define *page-width* 21cm)
(define *page-height* 29.7cm)
(define *left-margin* 2.5cm)
(define *right-margin* 2cm)
(define *top-margin* 3cm)
(define *bottom-margin* 2.7cm)
(define *header-margin* 1.3cm)
(define *footer-margin* 1.7cm)

;Default color space = RGB
(define *rgb-color-space* (color-space "ISO/IEC 10179:1996//Color-Space Family::Device RGB"))

; Set this variable to #t to activate the workaround for the keep-with-previous? bug in JadeTeX
; (which unfortunately exposes other JadeTeX problems with keep-with-next?...)
(define *keep-with-previous-bug-workaround* #f)



; ==========================
; FRENCH LANGUAGE DEFINITION
; ==========================

(declare-default-language (language 'fr 'fr))




; ==========================
; GENERAL PURPOSE PROCEDURES
; ==========================


; PROCEDURE get-standard-font:
; ----------------------------
; Returns the family name for the standard font, i.e the one for the "body" paragraph style
; if any, or the one for the first heading level

(define (get-standard-font)
  (let
    ( (style-info
        (if (assoc "body" *paragraph-styles*)
          (cadr (assoc "body" *paragraph-styles*))
          (cadr (assoc 1 *heading-styles*))
        )
      )
    )
    (caar style-info)
  )
)



; PROCEDURE section-level:
; ------------------------
; Returns the section nesting level for a node with type "Section"

(define (section-level node)
  ; Level is 1 + the parent's level, stopping whenever we hit a node with no parent or a node that is not a "Section"
  (cond
    ((null? node) 0)
    ((string-ci=? (gi node) "Section") (+ 1 (section-level (node-property 'parent node default: '()))))
    (#t 0)
  )
)



; ============================
; TABLE OF CONTENTS GENERATION
; ============================


; MODE toc:
; =========
; Used to generate the table of contents

(mode toc
  ; Only processed element is "Section", to create the corresponding ToC entry
  (element Section
    (let
      ((level (section-level (current-node))))
      
      (make sequence
        (make paragraph
          font-family-name: (get-standard-font)
          font-size: 12pt
          font-weight: (if (equal? level 1) 'bold 'medium)
          quadding: 'start
          start-indent: (* (- level 1) 0.5cm)
          space-before: 0.1cm
          hyphenate?: #t

          ; Only include text for actual title in ToC, not section number
          (process-matching-children "SectionTitle")
          (literal "  ")
          (make leader (literal "."))
          (make line-field field-align: 'end (current-node-page-number-sosofo))
        )
        (process-matching-children "Section")
      )
    )
  )
)



; ================
; INDEX GENERATION
; ================


; MODES index1 & index:
; =====================
; Used to generate the index. Only applied on "IndexEntry" elements

; First mode is for first page for an index entry
(mode index1
  (element IndexEntry
    (current-node-page-number-sosofo)
  )
)

; Second mode is for all other pages for an index entry, to add the leading ','
(mode index
  (element IndexEntry
    (make sequence
      (literal ", ")
      (current-node-page-number-sosofo)
    )
  )
)

; PROCEDURE index-mapping-append:
; -------------------------------
; Adds an entry in an index mapping
; - mapping: list of couples (entry text, list of "IndexEntry" nodes with this text)
; - entry: entry text for node to add
; - node: "IndexEntry" node to add
; Returns: the new mapping

(define (index-mapping-append mapping entry node)
  (cond
    ; If mapping is empty, build its single entry
    ((null? mapping) (list (list entry (list node))))
    ; If first element in mapping is the one for the added entry, just add the node to its list of nodes
    ((string=? (car (car mapping)) entry) (cons (list entry (cons node (cadr (car mapping)))) (cdr mapping)))
    ; If first element is not the one for the added entry, skip it and add entry to the remaining mapping
    (#t (cons (car mapping) (index-mapping-append (cdr mapping) entry node)))
  )
)

; PROCEDURE build-index-entry-mapping:
; ------------------------------------
; Builds a mapping index entry text -> list of "IndexEntry" nodes with this text
; - index-entry-mapping: The mapping built so far (list of couples (entry text, list of "IndexEntry" nodes)
; - nodes: list of "IndexEntry" nodes to record in mapping
; Returns: the built mapping

(define (build-index-entry-mapping index-entry-mapping nodes)
  (cond
    ; If no nodes to process, just return mapping in input
    ((node-list-empty? nodes) index-entry-mapping)
    ; If there are nodes to process
    (#t
      (let
        ; Get entry text for the first one
        ((index-entry-text (attribute-string "indexEntryText" (node-list-first nodes))))
        (build-index-entry-mapping
          ; Add this node to mapping
          (index-mapping-append index-entry-mapping index-entry-text (node-list-first nodes))
          ; And add nodes in the rest of the list
          (node-list-rest nodes)
        )
      )
    )
  )
)

; PROCEDURE split-mapping:
; ------------------------
; Part of the sorting process for index entry mappings; splits a mapping into 2 parts: the one
; for the index entries before a given one, and the one for those after it
; - mapping: The mapping to split (list of couples (entry text, list of "IndexEntry" nodes)
; - split-entry: The entry to split on
; Returns: a couple (part of mapping before split-entry, part of mapping after split-entry)

(define (split-mapping mapping split-entry)
  (cond
    ; If mapping is empty, both parts are empty
    ((null? mapping) (list mapping mapping))
    ; If mapping is not empty
    (#t
      (let
        ; Split its cdr
        ((parts (split-mapping (cdr mapping) split-entry)))
        
        (cond
          ; If first entry is before entry to split on, add it to first part
          ((string-ci<=? (car (car mapping)) split-entry) (list (cons (car mapping) (car parts)) (cadr parts)))
          ; If first entry is after entry to split on, add it to second part
          (#t (list (car parts) (cons (car mapping) (cadr parts))))
        )
      )
    )
  )
)

; PROCEDURE sort-index-entry-mapping:
; -----------------------------------
; Sorts an index entry mapping on the entry texts using the quick-sort algorithm
; - index-entry-mapping: The mapping to sort (list of couples (entry text, list of "IndexEntry" nodes)
; Returns: the sorted mapping

(define (sort-index-entry-mapping index-entry-mapping)
  (cond
    ; If mapping is empty, it's sorted
    ((null? index-entry-mapping) index-entry-mapping)
    ; If mapping is not empty
    (#t
      (let
        ; Extract first entry and split rest to get entries before the first and entries after it
        ((parts (split-mapping (cdr index-entry-mapping) (car (car index-entry-mapping)))))
        
        ; Sort both parts and add first entry in the middle
        (append (sort-index-entry-mapping (car parts)) (cons (car index-entry-mapping) (sort-index-entry-mapping (cadr parts))))
      )
    )
  )
)

; PROCEDURE build-index-entries:
; ------------------------------
; Builds all paragraphs for index entries in a mapping
; - index-sosofo: The sequence of flow objects where the index is built (sosofo)
; - index-entry-mapping: The sorted mapping continaing the entries (list of couples (entry text, list of "IndexEntry" nodes)
; - last-entry: last processed entry text
; Returns: the updated sosofo

(define (build-index-entries index-sosofo index-entry-mapping last-entry)
  (cond
    ; If mapping is empty, over
    ((null? index-entry-mapping) index-sosofo)
    ; If mapping is not empty
    (#t
      (let
        ( ; Get entry text and list of nodes for first element in mapping
          (entry (car (car index-entry-mapping)))
          (nodes (reverse (cadr (car index-entry-mapping))))
        )

        (build-index-entries
          ; Create paragraph for entry and append it to sequence
          (sosofo-append
            index-sosofo
            
            ; If entry starts with another letter than previous one, build index header
            (if (char-ci=? (string-ref entry 0) (string-ref last-entry 0))
              (empty-sosofo)
              (make paragraph
                font-family-name: (get-standard-font)
                font-size: 12pt
                font-weight: 'bold
                quadding: 'start
                space-before: 0.2cm
                keep-with-next?: #t
                
                (literal (string (char-upcase (string-ref entry 0))))
              )
            )
            
            (make paragraph
              font-family-name: (get-standard-font)
              font-size: 12pt
              quadding: 'start
              first-line-start-indent: -1.5cm
              start-indent: 1.5cm
              end-indent: 1cm

              ; Contents for paragraph is the entry text, a line of dots...
              (literal entry)
              (literal "  ")
              (make leader (literal "."))
              ; ... the first entry formatted as such, ...
              (with-mode index1 (process-node-list (car nodes)))
              ; ... and all following ones, formatted with a separator
              (with-mode index (process-node-list (apply node-list (cdr nodes))))
            )
          )
          ; Continue to build the paragraphs with the rest of the mapping
          (cdr index-entry-mapping)
          entry
        )
      )
    )
  )
)

; PROCEDURE make-index-entries:
; -----------------------------
; Builds the index for the document
; - nodes: The list of "IndexEntry" nodes in the whole document
; Returns: the sosofo for the index

(define (make-index-entries nodes)
  (let
    ; Build mapping entry text -> nodes for this text
    ((index-entries (build-index-entry-mapping '() nodes)))
    
    ; Sort mapping, then build the sosofo for the whole index
    (build-index-entries (make sequence (empty-sosofo)) (sort-index-entry-mapping index-entries) " ")
  )
)



; ================================
; REGULAR DOCUMENT BODY PROCESSING
; ================================


; PROCEDURE make-paragraph-with-contents:
; ---------------------------------------
; Creates a paragraph with a given style and contents
; - style-info: list of style information for paragraph, as in *heading-styles* or *paragraph-styles*
; - contents: sequence of flow objects for paragraph contents
; - force-keep-with-next?: used for optional workaround for bug in keep-with-previous? option
; Returns: the built paragraph

(define (make-paragraph-with-contents style-info contents #!key (force-keep-with-next? #f))
  (let
    ; Get style informations
    ( (font (list-ref style-info 0))
      (text-color (list-ref style-info 1))
      (align (list-ref style-info 2))
      (line-spacing (list-ref style-info 3))
      (left-margin (list-ref style-info 4))
      (right-margin (list-ref style-info 5))
      (first-indent (list-ref style-info 6))
      (space-above (list-ref style-info 8))
      (space-below (list-ref style-info 9))
      (page-break-before? (list-ref style-info 10))
      (keep-with-next?
        (if *keep-with-previous-bug-workaround*
          (or force-keep-with-next? (list-ref style-info 11))
          (list-ref style-info 11)
        )
      )
      (keep-with-previous? (list-ref style-info 12))
      (widow-orphan-lines (list-ref style-info 13))
    )
    
    ; Get font information and actual text color
    (let
      ( (font-family (list-ref font 0))
        (font-size (list-ref font 1))
        (bold (list-ref font 2))
        (italic (list-ref font 3))
        (underlined (list-ref font 4))
        (rgb-color (color *rgb-color-space* (car text-color) (cadr text-color) (caddr text-color)))
      )
      
      ; Build paragraph
      (make paragraph
        font-family-name: font-family
        font-size: font-size
        font-weight: (if bold 'bold 'medium)
        font-posture: (if italic 'italic 'upright)
        color: rgb-color
        quadding: align
        line-spacing: (+ (cadr font) line-spacing)
        start-indent: left-margin
        end-indent: right-margin
        first-line-start-indent: first-indent
        space-before: space-above
        space-after: space-below
        break-before: (if page-break-before? 'page #f)
        keep-with-next?: keep-with-next?
        keep-with-previous?: keep-with-previous?
        widow-count: widow-orphan-lines
        orphan-count: widow-orphan-lines
        hyphenate?: #t
        keep: #f

        (if underlined
          (make score type: 'after color: rgb-color contents)
          contents
        )
      )
    )
  )
)



; PROCEDURE make-paragraph:
; -------------------------
; Creates a paragraph for a node found in the input document
; - is-section?: If true, paragraph is a section header
; - style-info: List of style information for paragraph, as in *heading-styles* or *paragraph-styles*
; - force-keep-with-next?: used for optional workaround for bug in keep-with-previous? option
; Returns: the built paragraph

(define (make-paragraph is-section? style-info #!key (force-keep-with-next? #f))
  ; Get current node's children & header width
  (let
    ( (current-children (children (current-node)))
      (header-width (list-ref style-info 7))
    )
    
    (make-paragraph-with-contents
      style-info
      (make sequence
        ; If paragraph has a header width, create a line field
        (cond
          ((zero? header-width) (empty-sosofo))
          (#t
            (make line-field
              field-width: header-width

              (cond
                ; If paragraph is a section, header in is its SectionNumber child
                (is-section? (process-matching-children "SectionNumber"))
                ; If paragraph is not a section, header is in its first CharacterRange child
                (#t (process-node-list (node-list-first current-children)))
              )
            )
          )
        )
        (cond
          (is-section?
            ; Process paragraph body; for a section, it's its number if not yet done and its title
            (make sequence
              (cond
                ((zero? header-width) (process-matching-children "SectionNumber"))
                (#t                   (empty-sosofo))
              )
              (process-matching-children "SectionTitle")
            )
          )
          ; For a normal paragraph, include all contents except header if any
          (#t
            (cond
              ((zero? header-width) (process-children))
              (#t                   (process-node-list (node-list-rest current-children)))
            )
          )
        )
      )
      
      force-keep-with-next?: force-keep-with-next?
    )
  )
)


; ELEMENT IndexEntry:
; -------------------

(element IndexEntry
  ; Dummy line field for index entry so that an anchor can be created
  ; (For some reason, (make anchor) does not work...)
  (make line-field)
)


; ELEMENT CharacterRange:
; -----------------------

(element CharacterRange
  (let
    ; Get style information for character range
    ((style-info (assoc (attribute-string "style" (current-node)) *character-styles*)))
    
    (cond
      ; If there is, build a character sequence
      (style-info
        (let
          ( (font (car (cadr style-info)))
            (text-color (cadr (cadr style-info)))
          )
          
          (let
            ( (font-family (list-ref font 0))
              (font-size (list-ref font 1))
              (bold (list-ref font 2))
              (italic (list-ref font 3))
              (underlined (list-ref font 4))
              (rgb-color (color *rgb-color-space* (car text-color) (cadr text-color) (caddr text-color)))
            )

            (make sequence
              use: (style
                font-family-name: font-family
                font-size: font-size
                font-weight: (if bold 'bold 'medium)
                font-posture: (if italic 'italic 'upright)
                color: rgb-color
              )
              (if underlined
                (make score type: 'after color: rgb-color (process-children))
                (process-children)
              )
            )
          )
        )
      )
      ; If there isn't, process as usual
      (#t (process-children))
    )
  )
)


; ELEMENT Image:
; --------------

(element Image
  ; Compute reduction factor for image
  (let
    ( (image-width (string->number (attribute-string "width")))
      (image-height (string->number (attribute-string "height")))
      (min-reduction-factor (* (string->number (attribute-string "minReductionFactor")) 1pt))
      (max-width (- *page-width* (+ *left-margin* *right-margin*)))
      (max-height (- *page-height* (+ *top-margin* *bottom-margin*)))
    )
    (let
      ((factor (min min-reduction-factor (/ max-width image-width) (/ max-height image-height))))
      
      ; Create image in a centered paragraph
      (make paragraph
        quadding: 'center
        keep-with-previous?: #t   ; (Basically useless due to a bug in jadetex; keep it here just in case)
        keep-with-next?: #f       ; \
        break-before: #f          ;  | (Just in case...)
        keep: 'page               ; /
        
        (make external-graphic
          display?: #f
          max-width: (* image-width factor)
          max-height: (* image-height factor)
          entity-system-id: (attribute-string "file")
        )
        ; Children for image are index entries: process them
        (process-children)
      )
    )
  )
)


; ELEMENT Paragraph:
; ------------------

(element Paragraph
  (let
    ( ; Get style information for paragraph
      (style-info (assoc (attribute-string "style" (current-node)) *paragraph-styles*))
      ; Workaround for keep-with-previous? bug: figure out if paragraph is followed by an Image
      (image-follows?
        (let
          ((next-node (ifollow (current-node))))
          (if (node-list-empty? next-node) #f (string-ci=? (gi next-node) "Image"))
        )
      )
    )
    
    (cond
      ; If there is, build paragraph
      (style-info (make-paragraph #f (cadr style-info) force-keep-with-next?: image-follows?))
      ; If there isn't, do nothing
      (#t (empty-sosofo))
    )
  )
)


; ELEMENT Section:
; -----------------

(element Section
  (let
    ; Get children & style information for current section
    ( (current-children (children (current-node)))
      (style-info (assoc (section-level (current-node)) *heading-styles*))
    )
    
    (cond
      ; If there is
      (style-info
        (make sequence
          ; Build paragraph for section title
          (make-paragraph #t (cadr style-info))
          ; Process section contents
          (process-node-list (node-list-rest (node-list-rest current-children)))
        )
      )
      ; If there isn't, do nothing
      (#t (empty-sosofo))
    )
  )
)



; PROCEDURE make-left-header:
; ---------------------------
; Creates and returns the left header for all pages

(define (make-left-header #!key (right-gap 0pt))
  (make sequence
    use:
      (style
        font-family-name: (get-standard-font)
        font-size: 12pt
        color: (color *rgb-color-space* 0.0 0.0 0.0)
      )
    (literal *document-title*)
    (make paragraph-break)
    (make rule
      orientation: 'horizontal
      color: (color *rgb-color-space* 0.0 0.0 0.0)
      start-indent: 0pt
      end-indent: right-gap
    )
  )
)

; PROCEDURE make-right-header:
; ----------------------------
; Creates and returns the right header for all pages
; - right-gap: space to leave empty at the right of the header

(define (make-right-header #!key (right-gap 0pt))
  (make sequence
    (make external-graphic
      display?: #f

      max-width: 5.874cm
      max-height: 1.2cm
      entity-system-id: "logo.eps"
    )
    (make line-field field-width: right-gap (literal ""))
    (make paragraph-break)
  )
)

; PROCEDURE make-left-footer:
; ---------------------------
; Creates and returns the left footer for all pages
; - right-gap: space to leave empty at the right of the footer

(define (make-left-footer #!key (right-gap 0cm))
  (make sequence
    use:
      (style
        font-family-name: (get-standard-font)
        font-size: 12pt
        color: (color *rgb-color-space* 0.0 0.0 0.0)
      )
    (make rule
      orientation: 'horizontal
      color: (color *rgb-color-space* 0.0 0.0 0.0)
      start-indent: 0pt
      end-indent: right-gap
    )
    (make line-field (literal ""))
    (make paragraph-break)
    (literal *product-name*)
    (literal " ")
    (literal *product-version*)
  )
)

; PROCEDURE make-right-footer:
; ----------------------------
; Creates and returns the right footer for all pages
; - right-gap: space to leave empty at the right of the footer

(define (make-right-footer #!key (right-gap 0cm))
  (make sequence
    use:
      (style
        font-family-name: (get-standard-font)
        font-size: 12pt
        color: (color *rgb-color-space* 0.0 0.0 0.0)
      )
    (make line-field (literal ""))
    (make paragraph-break)
    (literal "Page ")
    (page-number-sosofo)
    (make line-field field-width: right-gap (literal ""))
  )
)


; ELEMENT Document:
; -----------------

(element Document
  (let
    ( ; Text for table of contents and index titles have the same style as first section headings
      (style-info (cadr (assoc 1 *heading-styles*)))
    )
    
    ; Pages are:
    (make sequence
      ; Page for document title
      (make simple-page-sequence
        page-width: *page-width*
        page-height: *page-height*
        left-margin: *left-margin*
        right-margin: *right-margin*
        top-margin: *header-margin*
        
        (make external-graphic
          display?: #t
          display-alignment: 'end
          max-width: 5.874cm
          max-height: 1.2cm
          entity-system-id: "logo.eps"
        )
        (make rule
          orientation: 'horizontal
          color: (color *rgb-color-space* 0.0 0.0 0.0)
          start-indent: 0pt
          end-indent: 0pt
          space-before: 0.1cm
          space-after: 0.5cm
        )
        (make paragraph
          font-family-name: (caar style-info)
          font-size: 24pt
          font-weight: 'bold
          quadding: 'end
          space-before: 1cm
          space-after: 1cm
          
          (literal *product-name*)
          (literal " ")
          (literal *product-version*)
        )
        (make paragraph
          font-family-name: (caar style-info)
          font-size: 24pt
          font-weight: 'bold
          quadding: 'end
          space-before: 1cm
          space-after: 1.5cm
          
          (literal *document-title*)
        )
        (let
          ((max-size (+ 1cm (- *page-width* (+ *left-margin* *right-margin*)))))
          
          (make external-graphic
            display?: #t
            display-alignment: 'start
            max-width: max-size
            max-height: max-size
            entity-system-id: "fond2.eps"
          )
        )
      )

      ; Pages for table of contents
      (make simple-page-sequence
        page-width: *page-width*
        page-height: *page-height*
        left-margin: *left-margin*
        right-margin: *right-margin*
        top-margin: *top-margin*
        bottom-margin: *bottom-margin*
        
        header-margin: *header-margin*
        left-header: (make-left-header)
        right-header: (make-right-header)

        footer-margin: *footer-margin*
        left-footer: (make-left-footer)
        right-footer: (make-right-footer)

        (make-paragraph-with-contents style-info (literal "Table of contents"))

        (with-mode toc (process-matching-children "Section"))
      )

      ; Pages for document itself
      (make simple-page-sequence
        page-width: *page-width*
        page-height: *page-height*
        left-margin: *left-margin*
        right-margin: *right-margin*
        top-margin: *top-margin*
        bottom-margin: *bottom-margin*
        
        header-margin: *header-margin*
        left-header: (make-left-header)
        right-header: (make-right-header)

        footer-margin: *footer-margin*
        left-footer: (make-left-footer)
        right-footer: (make-right-footer)

        (process-children)
      )

      ; Pages for index
      (make simple-page-sequence
        page-n-columns: 2         ; !!! OpenJade/JadeTeX extension; do not use with other DSSSL engines !!!
        page-width: *page-width*
        page-height: *page-height*
        left-margin: *left-margin*
        ; (The following is an ugly hack: since paragraph margin is used to create
        ;  the gutter between columns, decrease right page margin)
        right-margin: (- *right-margin* 1cm)
        top-margin: *top-margin*
        bottom-margin: *bottom-margin*
        
        ; (Ugly hack continued: make header and footer stop at 1cm from the right
        ;  page margin)
        header-margin: *header-margin*
        left-header: (make-left-header right-gap: 1cm)
        right-header: (make-right-header right-gap: 1cm)

        footer-margin: *footer-margin*
        left-footer: (make-left-footer right-gap: 1cm)
        right-footer: (make-right-footer right-gap: 1cm)

        (make-paragraph-with-contents style-info (literal "INDEX"))

        ; For index, process nodes with type "IndexEntry" through procedure defined above
        (make-index-entries (select-elements (descendants (current-node)) "IndexEntry"))
      )
    )
  )
)

</style-specification-body>
</style-specification>
</style-sheet>

