#ifndef _RTDS_TTCN_H_
#define _RTDS_TTCN_H_

#include "RTDS_BasicTypes.h"
#include "RTDS_CommonTypes.h"
#include "RTDS_Common.h"
#include "tci/tci.h"
#include "RTDS_String.h"

#include "trex/trex.h"

#include <stdio.h>
#include <stdarg.h>


/* enum type for value type (?, *, omit or a real value) */

typedef enum {
  VALUE,
  OMIT,
  ANYVALUE,
  ANYVALUEOROMIT,
  VALUEOROMIT
  } RTDS_TTCN_ValueType;
  
/* In TTCN-3, a special value can be passed as parameter of a template (?, *, omit). This case has to be tested dinamically in the generated C code */
/* For this, a C template is created, which has two attributes, the real value of the paramater, and its type (?, *, omit or value) */
/* In the classes generated for the template, it is this template which will be used for comparison instead of the simple value */
/* By doing this, "==" operator will be overload to always return true if the type of the value is ? or * */

template <typename T, typename U=T>
class Special 
	{
	public:
	// Default constructor: 
	// create a special object with default value
	Special()
		{
		this->_isRanged = FALSE;
    this->_rangeMin = 0;
    this->_rangeMax = 0;
    this->_isComplementRanged = FALSE;
		this->_complementRangeMin = 0;
		this->_complementRangeMax = 0;
		this->_optionLength = 0;
    this->_optionValue = NULL;
    this->_complementLength = 0;
    this->_complementValue = NULL;
		this->_valueType = VALUE;
		}
	// Value constructor: 
	// create a special object with an actual value (for VALUE)
	Special(T value)
		{
		this->_isRanged = FALSE;
    this->_rangeMin = 0;
    this->_rangeMax = 0;
    this->_isComplementRanged = FALSE;
		this->_complementRangeMin = 0;
		this->_complementRangeMax = 0;
		this->_optionLength = 0;
    this->_optionValue = NULL;
    this->_complementLength = 0;
    this->_complementValue = NULL;
		this->_value = value;
		this->_valueType = VALUE;
		}
	// Type constructor: 
	// create a special object for either OMIT, ANYVALUE, or ANYVALUEOROMIT
	Special(RTDS_TTCN_ValueType valueType)
		{
		this->_isRanged = FALSE;
    this->_rangeMin = 0;
    this->_rangeMax = 0;
    this->_isComplementRanged = FALSE;
		this->_complementRangeMin = 0;
		this->_complementRangeMax = 0;
		this->_optionLength = 0;
    this->_optionValue = NULL;
    this->_complementLength = 0;
    this->_complementValue = NULL;
		this->_valueType = valueType;
		}
	// Destructor:
  // free option values array if set
	~Special()
	  {
	  if (this->_optionLength > 0)
	    {
	    free(this->_optionValue);
	    }
    if (this->_complementLength > 0)
	    {
	    free(this->_complementValue);
	    }
	  }
	// Conversion operator: 
	// a special object is implicitly converted to the actual value
	operator T& ()
		{
		return this->_value;
		}
  // Address operator: 
	// the adress of a special object is implicitly converted to the address of the actual value
	T* operator & ()
		{
		return &(this->_value);
		}
	// Function call operator: 
	// a special object followed by empty parenthesis () is converted to the actual value
	T& operator () ()
	  {
	  return this->_value;
	  }
	// Member access operator: 
	// allow access of members of the actual value directly from the special object
	T& operator -> ()
		{
		return this->_value;
		}
	// Assignment operator:
	// assign directly to the actual value through the special object
	T& operator = (T value) 
		{
		this->_value = value;
		this->_valueType = VALUE;
		return this->_value;
		}
  
  // Equality operator:
  // if there is an actual value, check equality on it
  // otherwise, comparison to OMIT, ANYVALUE, or ANYVALUEOROMIT returns always true
	RTDS_BOOLEAN operator == (T value)
		{
    switch (this->_valueType)
		  {
			case ANYVALUE: return TRUE;
			case ANYVALUEOROMIT: return TRUE;
			case VALUEOROMIT:
			  // In case of a special comparison (range or multi possible value)
        if ( ( this->_isRanged == TRUE ) || (this->_optionLength > 0) || ( this->_isComplementRanged == TRUE ) || ( this->_complementLength > 0 ))
          {  
          // In case of a range, test if the value is inside the range
          if (this->_isRanged == TRUE)
            {
            if (value > this->_rangeMax) 
              {
              return FALSE;
              }
            if (value < this->_rangeMin) 
              {
              return FALSE;
              }
            }
          // In case of multi-option, test if the value is in the array of options
          if (this->_optionLength > 0)
            {
            for (int i=0; i<this->_optionLength; ++i)
              {
              if (this->_optionValue[i] == value)
                {
                return TRUE;
                }
              }
            return FALSE;
            }
          if (this->_isComplementRanged == TRUE)
            {
				    if (value < this->_complementRangeMax && value > this->_complementRangeMin) 
				      {
				      return FALSE;
				      }
            }
          // In case of complement, test if the value is not in the array of complement
          if (this->_complementLength > 0)
            {
            for (int i=0; i<this->_complementLength; ++i)
              {
              if (this->_complementValue[i] == value)
                {
                return FALSE;
                }
              }
            }
				  return TRUE;
	        }
        else
			    {
				  return (RTDS_BOOLEAN)(this->_value == value);
				  }
	    case VALUE:
			  // In case of a special comparison (range or multi possible value)
        if ( ( this->_isRanged == TRUE ) || (this->_optionLength > 0) || ( this->_isComplementRanged == TRUE ) || ( this->_complementLength > 0 ))
          {  
          // In case of a range, test if the value is inside the range
          if (this->_isRanged == TRUE)
            {
            if (value > this->_rangeMax) 
              {
              return FALSE;
              }
            if (value < this->_rangeMin) 
              {
              return FALSE;
              }
            }
          // In case of multi-option, test if the value is in the array of options
          if (this->_optionLength > 0)
            {
            for (int i=0; i<this->_optionLength; ++i)
              {
              if (this->_optionValue[i] == value)
                {
                return TRUE;
                }
              }
            return FALSE;
            }
          if (this->_isComplementRanged == TRUE)
            {
				    if (value < this->_complementRangeMax && value > this->_complementRangeMin) 
				      {
				      return FALSE;
				      }
            }
          // In case of complement, test if the value is not in the array of complement
          if (this->_complementLength > 0)
            {
            for (int i=0; i<this->_complementLength; ++i)
              {
              if (this->_complementValue[i] == value)
                {
                return FALSE;
                }
              }
            }
				  return TRUE;
	        }
        else
			    {
				  return (RTDS_BOOLEAN)(this->_value == value);
				  }
			default: return (RTDS_BOOLEAN)(this->_value == value);
			}
		}	
	
  // Setter for value type
	void setValueType(RTDS_TTCN_ValueType valueType)
		{
		this->_valueType = valueType;
		}
	
  void setRangeValue(int minValue, int maxValue)
		{
		this->_isRanged = TRUE;
		this->_rangeMin = minValue;
		this->_rangeMax = maxValue;
		}
	
  void setRangeComplementValue(int minValue, int maxValue)
		{
		this->_isComplementRanged = TRUE;
		this->_complementRangeMin = minValue;
		this->_complementRangeMax = maxValue;
		}
  
  void setOptionValue(int optionLength, ...)
	  {
	  this->_optionLength = optionLength;
	  this->_optionValue = (T *) malloc(optionLength * sizeof(T));
	  va_list args;
    va_start(args, optionLength);
    for (int i=0; i<optionLength; ++i)
      {
      this->_optionValue[i] = (T)va_arg(args, U);
      }
    va_end(args);
	  }
    
  void setComplementValue(int complementLength, ...)
	  {
    RTDS_BOOLEAN toto;
	  this->_complementLength = complementLength;
    this->_complementValue = (T *) malloc(complementLength * sizeof(T));
	  va_list args;
    va_start(args, complementLength);
    for (int i=0; i<complementLength; ++i)
      {
      this->_complementValue[i] = (T)va_arg(args, U);
      }
    va_end(args);
	  }
	
  void setAllOptions(Special<T,U>& SpecialVar)
    {
    this->_valueType = SpecialVar._valueType;
    this->_isRanged = SpecialVar._isRanged;
		this->_rangeMin = SpecialVar._rangeMin;
		this->_rangeMax = SpecialVar._rangeMax;
    this->_isComplementRanged = SpecialVar._isComplementRanged;
    this->_complementRangeMin = SpecialVar._complementRangeMin;
		this->_complementRangeMax = SpecialVar._complementRangeMax;
    this->_optionLength = SpecialVar._optionLength;
    this->_complementLength = SpecialVar._complementLength;
    }
  
  // Getter for value type
	RTDS_TTCN_ValueType getValueType()
		{
		return this->_valueType;
		}
  
  int getOptionLength()
		{
		return this->_optionLength;
		}
  
  Special<T,U>* getSpecialObject()
    {
    return this;
    }
	
  RTDS_BOOLEAN isRanged()
    {
    return this->_isRanged;
    }
  
  int getRangeMin()
    {
    return this->_rangeMin;
    }
  
  int getRangeMax()
    {
    return this->_rangeMax;
    }
  
  RTDS_BOOLEAN isComplementRanged()
    {
    return this->_isComplementRanged;
    }
    
  int getComplementRangeMin()
    {
    return this->_complementRangeMin;
    }
  
  int getComplementRangeMax()
    {
    return this->_complementRangeMax;
    }
  
	private:
  RTDS_BOOLEAN _isRanged;          // True in case of a range in a template
  int _rangeMin;                   // In case of range in a template, store the min value of the range
	int _rangeMax;                   // In case of range in a template, store the max value of the range
  
  RTDS_BOOLEAN _isComplementRanged;           // True in case of a complement range in a template
  int _complementRangeMin;                    // In case of complement range in a template, store the min value of the range
	int _complementRangeMax;                    // In case of complement range in a template, store the max value of the range
  
  int _optionLength;               // In case of multi-option value in a template, store the number of possible options 
                                  // e.g., template myType t := { f1 := 42, f2 := (5, 7) } where f2 value is either 5 or 7 (integer options)
  T * _optionValue;                // In case of multi-option value in a template, store the possible options in an array
	
  int _complementLength;            // In case of complement, store the number of complemented elements
  T * _complementValue;             // In case of complement, store the complemented elements
	
  
  T _value;                        // Actual value (if any) encapsulated in the special object
	RTDS_TTCN_ValueType _valueType;  // Value type of special object
	};


class RTDS_TTCN_Template
  {
  };

// Declaration of all Template class for basic types

class RTDS_TTCN_Templateint : public RTDS_TTCN_Template
  {
  public:
  RTDS_TTCN_Templateint() 
    { 
    this->_value = 0; 
    }
  virtual RTDS_BOOLEAN match(int variableToMatch, RTDS_String BackTrace_Template)
    { 
    return (RTDS_BOOLEAN) (this->_value == variableToMatch); 
    }
  virtual int valueof()
    { 
    return this->_value; 
    }
  public:
  int _value;
  };  
  
class RTDS_TTCN_TemplateRTDS_String : public RTDS_TTCN_Template
  {
  public:
  RTDS_TTCN_TemplateRTDS_String() 
    { 
    RTDS_StringAssign(this->_value, ""); 
    }
  virtual RTDS_BOOLEAN match(char * variableToMatch, RTDS_String BackTrace_Template)
    { 
    return (RTDS_BOOLEAN) (strcmp(this->_value, variableToMatch) == 0); 
    }
  virtual char * valueof()
    { 
    return this->_value;
    }
  public:
  RTDS_String _value; 
  };
  
class RTDS_TTCN_TemplateRTDS_BOOLEAN : public RTDS_TTCN_Template
  {
  public:
  RTDS_TTCN_TemplateRTDS_BOOLEAN()
    {
    this->_value = FALSE;
    }
  virtual RTDS_BOOLEAN match(RTDS_BOOLEAN variableToMatch, RTDS_String BackTrace_Template)
    { 
    return (RTDS_BOOLEAN) (this->_value == variableToMatch); 
    }
  virtual RTDS_BOOLEAN valueof()
    {
    return this->_value;
    }
  public:
  RTDS_BOOLEAN _value;
  };
  
class RTDS_TTCN_Templatedouble : public RTDS_TTCN_Template
  {
  public:
  RTDS_TTCN_Templatedouble()
    {
    this->_value = 0.0;
    }
  virtual RTDS_BOOLEAN match(double variableToMatch, RTDS_String BackTrace_Template)
    { 
    return (RTDS_BOOLEAN) (this->_value == variableToMatch); 
    }
  virtual double valueof()
    {
    return this->_value;
    }
  public:
  double _value;
  };
  
class RTDS_TTCN_Templateverdicttype : public RTDS_TTCN_Template
  {
  public:
  RTDS_TTCN_Templateverdicttype()
    {
    this->_value = TCI_VERDICT_NONE;
    }
  virtual RTDS_BOOLEAN match(verdicttype variableToMatch, RTDS_String BackTrace_Template)
    { 
    return (RTDS_BOOLEAN) (this->_value == variableToMatch); 
    }
  virtual verdicttype valueof()
    {
    return this->_value;
    }
  public:
  verdicttype _value;
  };  

  
class RTDS_TTCN_TemplatePattern
  {
  public:
  RTDS_TTCN_TemplatePattern(char * patternToMatch)
		{
		RTDS_StringAssign(this->patternToMatch, patternToMatch);
		}
  RTDS_BOOLEAN match(char * variableToMatch)
    {
		const char *regexError = NULL;
		TRex *regexObj = trex_compile(patternToMatch, &regexError);
		if (regexObj)
			{
			RTDS_BOOLEAN regexMatchResult = (RTDS_BOOLEAN) trex_match(regexObj, (const char *) variableToMatch);
			trex_free(regexObj);
			return regexMatchResult;
			}
    return FALSE;
    }
  char * valueof()
    {
    return patternToMatch;
    }
  public :
  RTDS_String patternToMatch;
  };

typedef enum
  {
  RTDS_MAP      = 0,
  RTDS_CONNECT  = 1,
  RTDS_SYS      = 2   // for system and timer
  } RTDS_TTCN_ConnectionType;
  
typedef struct RTDS_TTCN_TestcaseParameters
  {
  QualifiedName testcaseName;
  TciParameterTypeListType parametersTypeList;
  }RTDS_TTCN_TestcaseParameters;
  
typedef struct RTDS_TTCN_PortIdList
  {
  TriPortId                     * portId;
  struct RTDS_TTCN_PortIdList   * next;
  } RTDS_TTCN_PortIdList;


typedef struct RTDS_TTCN_TimerInfo
  {
  long              timerID;
  int               timerNumber;
  unsigned long     timerUniqueID;
  RTDS_String       timerName;
  TriTimerDuration  duration;
  TriTimerDuration  timeoutValue;
  TimerStatusType   timerStatus;
  } RTDS_TTCN_TimerInfo;


typedef struct RTDS_TTCN_TimerInfoList
  {
  RTDS_TTCN_TimerInfo               * timerInfo;
  struct RTDS_TTCN_TimerInfoList    * next;
  } RTDS_TTCN_TimerInfoList;

typedef RTDS_BOOLEAN (*RTDS_TTCN_functionpointer)(char *, char **, int);
  
typedef struct RTDS_TTCN_ActivateAlstepList
  {
  RTDS_TTCN_functionpointer 			      altstepfunction;
  char 									              * altstepName;
  struct RTDS_TTCN_ActivateAlstepList * next;
  } RTDS_TTCN_ActivateAlstepList;
  
  
typedef struct RTDS_TTCN_PortMappingInfo
  {
  RTDS_RtosQueueId                    queueControlBlock;
  RTDS_MessageHeader                * currentMessage;
  RTDS_TTCN_ConnectionType            connectionType;
  PortStatusType                      portStatus;
  TriPortId                         * portId;
  RTDS_TTCN_PortIdList              * mappedPort;
  struct RTDS_TTCN_PortMappingInfo  * next;
  } RTDS_TTCN_PortMappingInfo;
  
  
typedef struct RTDS_TTCN_Template_Pool
  {
  RTDS_TTCN_Template        * templateID;
  RTDS_TTCN_Template_Pool   * next;
  } RTDS_TTCN_Template_Pool;
  
/* Global pool for all used template */
extern RTDS_TTCN_Template_Pool * RTDS_TTCN_globalTemplatePool;
  
typedef struct RTDS_TTCN_GlobalComponentInfo
  {
  TriComponentId                        * componentID;
  TciTestComponentKindType                componentKindType;
  ComponentStatusType                     componentStatus;
  verdicttype                             componentVerdict;
  RTDS_String						                  testcaseName;
  RTDS_GlobalProcessInfo                * RTDS_currentContext;
  RTDS_TTCN_PortMappingInfo             * portMappingInfo;
  RTDS_TTCN_TimerInfoList               * timerInfoList;
  RTDS_TTCN_ActivateAlstepList			    * activateAltstepList;
  void                                  * cmpVariables;
  struct RTDS_TTCN_GlobalComponentInfo  * next;
  } RTDS_TTCN_GlobalComponentInfo;
  

typedef RTDS_TTCN_TimerInfo * RTDS_TIMER;

/* Global list for component info */
extern RTDS_TTCN_GlobalComponentInfo * RTDS_TTCN_globalComponentInfo;

typedef struct RTDS_TTCN_GlobalComplexVariableElementListType
  {
  char														* ComplexVariableElement;
  char														* UsedPool;
  RTDS_BOOLEAN												isConstant;
  struct RTDS_TTCN_GlobalComplexVariableElementListType 	* next;
  } RTDS_TTCN_GlobalComplexVariableElementListType;

/* Global list for complex variable */
extern RTDS_TTCN_GlobalComplexVariableElementListType * RTDS_TTCN_globalComplexVariableElementListType;
  
  
/* Type for a port in the system interface */
typedef struct _RTDS_TTCN_SystemInterfacePort
  {
  char                                  * portName;
  RTDS_RtosQueueId                        portMessageQueue;
  struct _RTDS_TTCN_SystemInterfacePort * next;
  } RTDS_TTCN_SystemInterfacePort;

/* Global variable for all ports in system interface */
extern RTDS_TTCN_SystemInterfacePort * RTDS_TTCN_systemInterfacePorts;

/* Types for module and testcase encoded parameters */
typedef struct _RTDS_TTCN_EncDecParameter
  {
  char                                * name;
  char                                * value;
  struct _RTDS_TTCN_EncDecParameter   * next;
  } RTDS_TTCN_EncDecParameter;

typedef struct _RTDS_TTCN_EncDecTestcase
  {
  char                              * name;
  RTDS_TTCN_EncDecParameter         * parameters;
  struct _RTDS_TTCN_EncDecTestcase  * next;
  } RTDS_TTCN_EncDecTestcase;

typedef struct _RTDS_TTCN_EncDecModule
  {
  char                            * name;
  RTDS_TTCN_EncDecParameter       * parameters;
  RTDS_TTCN_EncDecTestcase        * testcases;
  struct _RTDS_TTCN_EncDecModule  * next;
  } RTDS_TTCN_EncDecModule;
  
typedef struct _RTDS_TTCN_EncDecConfiguration
  {
  char                                  * name;
  RTDS_TTCN_EncDecModule                * modules;
  struct _RTDS_TTCN_EncDecConfiguration * next;
  } RTDS_TTCN_EncDecConfiguration;

/* Global list of all configurations with module and testcase encoded parameters */
extern RTDS_TTCN_EncDecConfiguration * RTDS_TTCN_encdecConfigurations;


#ifdef __cplusplus
extern "C" {
#endif

// Function prototypes

void RTDS_TTCN_Init(void);
void RTDS_TTCN_Postamble(void);


/******************************************************************
 *        TEMPLATE FUNCTIONS
 * ----------------------------------------------------------------
 *    Functions for template operations
 *
 *******************************************************************/

RTDS_TTCN_Template * RTDS_TTCN_RegisterTemplatePool( RTDS_TTCN_Template * RTDS_New_Template);


/******************************************************************
 *        STATUS FUNCTIONS
 * ----------------------------------------------------------------
 *    Functions call to check component status
 *
 *******************************************************************/

void RTDS_TTCN_SetStatus( RTDS_SdlInstanceId * Component2Set, ComponentStatusType status );
int RTDS_TTCN_CheckAlive( RTDS_SdlInstanceId * Component2test );
int RTDS_TTCN_CheckRunning( RTDS_SdlInstanceId * Component2test );
int RTDS_TTCN_CheckStopped( RTDS_SdlInstanceId * Component2test );
int RTDS_TTCN_CheckKilled( RTDS_SdlInstanceId * Component2test );


/******************************************************************
 *        TIMER FUNCTIONS
 * ----------------------------------------------------------------
 *    Functions call to check timer status
 *
 *******************************************************************/

int RTDS_TTCN_CheckAnyTimerRunning (RTDS_TTCN_GlobalComponentInfo * );
int RTDS_TTCN_CheckTimeout (RTDS_TIMER);
RTDS_TIMER RTDS_TTCN_TimerCreate (long, int, char*, TriTimerDuration);


/******************************************************************
 *        MESSAGE FUNCTIONS
 * ----------------------------------------------------------------
 *    Functions for message operations
 *
 *******************************************************************/

void RTDS_TTCN_UpdateFirstMessage ( RTDS_TTCN_GlobalComponentInfo * RTDS_TTCNcurrentComponentInfo);
void RTDS_TTCN_UpdateTimerLists ( RTDS_TTCN_GlobalComponentInfo * RTDS_currentContext );
int RTDS_TTCN_MsgQueueCheck( char *, RTDS_MessageHeader *, RTDS_TTCN_GlobalComponentInfo *);


/******************************************************************
 *        TESTCASES FUNCTIONS
 * ----------------------------------------------------------------
 *    Functions for testcase execution and status
 *
 *******************************************************************/

void RTDS_TTCN_TestcaseTerminated(RTDS_GlobalProcessInfo * RTDS_currentContext, verdicttype verdict);
verdicttype RTDS_TTCN_GetVerdict(RTDS_SdlInstanceId * Component2Get);
verdicttype RTDS_TTCN_GetFinalVerdict();
char * RTDS_TTCN_getcurrentTestcase();

RTDS_TTCN_ActivateAlstepList * RTDS_TTCN_activateAltstep(char *, RTDS_TTCN_functionpointer,RTDS_TTCN_GlobalComponentInfo*);

/******************************************************************
 *        predefined FUNCTIONS
 * ----------------------------------------------------------------
 *    predefined functions in TTCN-3
 *
 *******************************************************************/

char * RTDS_TTCN_int2str(int, char *);

RTDS_BitString * RTDS_TTCN_int2bit(int, int, RTDS_BitString * );

RTDS_OctetString * RTDS_TTCN_int2oct(int, int, RTDS_OctetString * );


/******************************************************************
 *        PARAMETER FUNCTIONS
 * ----------------------------------------------------------------
 *    Functions for module and testcase parameters
 *
 *******************************************************************/
 
RTDS_TTCN_EncDecConfiguration * RTDS_TTCN_GetEncDecConfiguration(char *, RTDS_BOOLEAN);
RTDS_TTCN_EncDecModule * RTDS_TTCN_GetEncDecModule(RTDS_TTCN_EncDecConfiguration *, char *, RTDS_BOOLEAN);
RTDS_TTCN_EncDecTestcase * RTDS_TTCN_GetEncDecTestcase(RTDS_TTCN_EncDecModule *, char *, RTDS_BOOLEAN);
void RTDS_TTCN_AddEncDecParameter(RTDS_TTCN_EncDecModule *, RTDS_TTCN_EncDecTestcase *, char *, char *);
RTDS_TTCN_EncDecParameter * RTDS_TTCN_GetEncDecParameter(RTDS_TTCN_EncDecModule *, RTDS_TTCN_EncDecTestcase *, char *);
void RTDS_TTCN_ClearEncDecParameters();

#ifdef __cplusplus
}
#endif

#endif
