/***********************************************************************************
 *						RTDS FORMAT TRACE
 * ---------------------------------------------------------------------------------
 *	Fonctions for tracing and dynamically configure trace visibility :
 *		RTDS_FormatTrace
 *			RTDS_ConfigureTrace
 ***********************************************************************************/

#ifdef RTDS_FORMAT_TRACE

#include <stdio.h> /* for sprintf */
#include <string.h> /* for strcat */

#include "RTDS_FormatTrace.h"

const char commandEnum[17][20]={	"messageSent| ", "messageReceived| ", "timerStarted| ", "timerCancelled| ", "taskCreated| ","taskDeleted| ", "takeAttempt| ", "takeSucceeded| ", "giveSem| ", "taskChangedState| ", "initDone| ", "takeTimedOut| ", "systemError| ", "semaphoreCreated| ", "semaphoreDeleted| ", "messageSaved| ", "information| "};
const char timerTimedOutCmd[17]= "timerTimedOut| ";

/* Global variable used for configuration */
int RTDS_globalEventTraceEnable = 1;
int RTDS_globalEventTimeEnable = 1;
int RTDS_globalEventMessageDataEnable = 1;
int RTDS_globalEventMessageEnable = 1;
int RTDS_globalEventTimerEnable = 1;
int RTDS_globalEventSemaphoreEnable = 1;
int RTDS_globalEventProcessEnable = 1;
int RTDS_globalEventSdlStateEnable = 1;

/******************************************************************************************
 *								 FUNCTION 	RTDS_ConfigureTrace(...)
 *-------------------------------------------------------------------
 *	Purpose :Configure the dynamic trace format according to the input configuration struct
 *
 ******************************************************************************************/

void RTDS_ConfigureTrace(RTDS_CmdTraceConfig * configStruc)
	{
	int state = 1;

	state = configStruc->status;

	switch (configStruc->configId)
		{
		case TRACING:
			if (state)
				{RTDS_DTRACE_RESUME();}
			else
				{RTDS_DTRACE_PAUSE();}
			break;
		case ALL_SDLEVENT:
			RTDS_DTRACE_MESSAGE(state);
			RTDS_DTRACE_TIMER(state);
			RTDS_DTRACE_SDLSTATE(state);
			RTDS_DTRACE_PROCESS(state);
			RTDS_DTRACE_SEMAPHORE(state);
			break;
		case SDL_MSG:
			RTDS_DTRACE_MESSAGE(state);
			break;
		case SDL_MSG_DATA:
			RTDS_DTRACE_MESSAGEDATA_INFO(state);
			break;
		case SDL_TIMER:
			RTDS_DTRACE_TIMER(state);
			break;
		case SDL_PROCESS:
			RTDS_DTRACE_PROCESS(state);
			break;
		case SDL_SEMAPHORE:
			RTDS_DTRACE_SEMAPHORE(state);
			break;
		case SDL_STATE:
			RTDS_DTRACE_SDLSTATE(state);
			break;
		case SDL_CLOCK:
			RTDS_DTRACE_CLOCK_INFORMATION(state);
			break;
		}
	}

/*
 * *****************************************************************************
 *								 FUNCTION  RTDS_FormatTrace(...)
 * -------------------------------------------------------------------
 *	Purpose : Generates a text string out of the trace information
 *		to be sent to the MSC Tracer.
 * *****************************************************************************
 *							SDL EVENT :
 * Param:
 *	 unsigned long				 sysTime: 			system time
 *	 RTDS_GlobalTraceInfo  traceEntry:		trace to format
 *	 int									 waitAckMode: 	RTDS_DTRACE_ACK_WAIT if waiting for ack, RTDS_DTRACE_ACK_NOWAIT otherwise
 *	 char **							 formatedTrace: resulting string allocated in the function
 *
 * Returns:
 *	 int: RTDS_OK or RTDS_ERROR
 *	 In some cases the *formatedTrace return value might null.
 *
 * *****************************************************************************
 */
int RTDS_FormatTrace(
		unsigned long time,
		RTDS_GlobalTraceInfo traceEntry,
		int waitAckMode,
		char **formatedTrace)
	{
	/* Variable for formatting trace information */
/*
	char commandEnum[16][20]={	"messageSent| ", "messageReceived| ", "timerStarted| ", "timerCancelled| ", "taskCreated| ","taskDeleted| ", "takeAttempt| ", "takeSucceeded| ", "giveSem| ", "taskChangedState| ", "initDone| ", "takeTimedOut| ", "systemError| ", "semaphoreCreated| ", "semaphoreDeleted| ", "messageSaved| "};
	char timerTimedOutCmd[17]= "timerTimedOut| ";
*/
	int regularEventTraceSize=256;
	char regularParameter[RTDS_REGULAR_PARAMSIZE];
	int ignoreCurrentCommand = 1;
	char * commandString=NULL;

	/* Event information */
	int event = (int)(traceEntry.event);
	void * eventParam1 = (void *)(traceEntry.eventParameter1);
	long eventParam2 = (long)(traceEntry.eventParameter2);
	RTDS_GlobalProcessInfo *currentContext = traceEntry.currentContext;

	/* Generic information*/
	unsigned long pId=0;
	int pNumber = 0;
	void * errorId = NULL;

	/* Process event Variable*/
	RTDS_GlobalProcessInfo * tmpProcInfo=NULL;

	/* Message event Variable*/
	#if defined(RTDS_SIMULATOR)
		unsigned long mId;
	#endif
	unsigned long sigNum, msgName;
	unsigned char * msgData=NULL;
	RTDS_MessageHeader *currentMessage=NULL;
	char * structuredData=NULL, * extendedCommandString=NULL;

	/* Timer event Variable*/
	unsigned long timerUniqueId;
	unsigned long timeoutValue=0;

	/* Semaphore event Variable*/
	unsigned long semaphoreId;
	int semNumber=0;
	RTDS_GlobalSemaphoreInfo *semInfo;
	long stateName=-1;
	int i;
	long stillAvailable=0;

		
	/* Init formatted string */
	*formatedTrace = NULL;

	#ifdef RTDS_FORMAT_TRACE
	#ifndef RTDS_SIMULATOR
		/* ignore initDone Event if not using RTDS */
		if (event == RTDS_initDone)
			return RTDS_OK;
	#endif /* RTDS_SIMULATOR */
	#endif /* RTDS_FORMAT_TRACE */

	/* Change the regular Event Trace Size if needed */
	/* for instance size is changed only when message event trace is set to print message data */
	if(RTDS_globalEventMessageDataEnable && (event == RTDS_messageSent || event == RTDS_messageReceived ))
		{
		regularEventTraceSize = (((RTDS_MessageHeader*)eventParam1)->dataLength)*3 + regularEventTraceSize;
		}

	/* All events require currentContext */
	if (currentContext != NULL && pId == 0)
		{
		pNumber = currentContext->sdlProcessNumber;
		pId = (unsigned long)currentContext->mySdlInstanceId;
		}
	/* Test visibility and if trace is enable */
	/* pNumber = 0 and pId = 0 if RTDS_Start*/
	if (pNumber != 0)
		if (!RTDS_globalEventTraceEnable)
			{
			*formatedTrace = NULL;
			return RTDS_OK;
			}

	/* Init command string */
	commandString = (char *)RTDS_MALLOC(regularEventTraceSize);
	for (i=0; i<regularEventTraceSize; i++)
		commandString[i] = '\0';

	/* ADD COMMAND */
	if (event != RTDS_messageReceived)
		{
		strcat(commandString, commandEnum[event]);
		}
	else
		{
		/* case of RTDS_messageReceived event : could be regular message or timer timed out*/
		if (((RTDS_MessageHeader *)eventParam1)->timerUniqueId)
			strcat(commandString, timerTimedOutCmd);
		else
			strcat(commandString, commandEnum[event]);
		}

	/* ADD GENERIC OPTIONS*/
	/* Add time information option */
	if(RTDS_globalEventTimeEnable)
		{
		sprintf(regularParameter,"-t| %lu| ", time);
		strcat(commandString, regularParameter);
		}

	/* Add process Name info option */
	/* pNumber = 0 and pId = 0 if RTDS_Start*/
	if (pNumber != 0)
		{
		if (event == RTDS_semaphoreCreated || event == RTDS_semaphoreDeleted || event == RTDS_processCreated)
			sprintf(regularParameter,"-B| %d| ", pNumber);
		else
			sprintf(regularParameter,"-b| %d| ", pNumber);
		strcat(commandString, regularParameter);
		}

	/* ADD SPECIFIC OPTIONS*/
	switch(event)
		{
		case RTDS_processCreated:  /* command : "taskCreated [-t time] [-n pName]  [-b pNumber] [-c creatorId] [-N creatorName] [-B creatorNumber] pId" 	eventParam1 = newProcessInfo->processId and eventParam2 = NULL */
			/* Add [-n pName]  [-b pNumber] option */
			tmpProcInfo = (RTDS_GlobalProcessInfo*)eventParam1;
			if (tmpProcInfo->sdlProcessNumber != 0)
				{
				sprintf(regularParameter,"-b| %d| ", tmpProcInfo->sdlProcessNumber);
				strcat(commandString, regularParameter);
				}
			/* Add [-c creatorId]*/
			if(pId != 0)
				{
				sprintf(regularParameter,"-c| 0x%lx| ", pId);
				strcat(commandString, regularParameter);
				}
			/* Change pId parameter */
			pId = (unsigned long)((RTDS_GlobalProcessInfo*)eventParam1)->mySdlInstanceId;
		case RTDS_processDied:			   /* command : "taskDeleted [-t time] [-n pName]  [-b pNumber] 								pId"	 eventParam1 = newProcessInfo->processId and eventParam2 = NULL */
			if (!RTDS_globalEventProcessEnable) break;
			ignoreCurrentCommand = 0;
			break;

		case RTDS_sdlStateSet:		    /* command : "taskChangedState  [-t time] [-n pName]  [-b pNumber] pId stateName" eventParam1 = RTDS_currentContext->processId and eventParam2 = NEW_SDL_STATE */
			if (!RTDS_globalEventSdlStateEnable) break;
			stateName = (long)eventParam2;
			ignoreCurrentCommand = 0;
			break;

		case RTDS_messageSent:			   /* command : "messageSent		 [-t time] [-n pName] [-b pNumber] [-i mId] [-d msgData] pId msgName sigNum"		eventParam1 = &RTDS_messageToSend and eventParam2 = receiver */
			if ( waitAckMode == RTDS_DTRACE_ACK_NOWAIT )
				{
				sprintf(regularParameter,"-k| ");
				strcat(commandString, regularParameter);
				}
		case RTDS_messageReceived:	 /* command : "messageReceived	 [-t time] [-n pName] [-b pNumber] [-i mId] [-d msgData] pId msgName sigNum"		eventParam1 = RTDS_currentContext->currentMessage and eventParam2 = SELF command : "timerTimedOut 	[-t time] [-n pName] [-b pNumber] 			 [-T timerName] pId tId" */
		case RTDS_messageSaved: 		  /* command : "messageSaved     [-t time] [-n pName] [-b pNumber] [-i mId]	       						pId msgName sigNum" 	eventParam1 = CURRENT_MESSAGE and eventParam2 = SELF */
			if (!RTDS_globalEventMessageEnable) break;

			timerUniqueId = ((RTDS_MessageHeader *)eventParam1)->timerUniqueId;
			currentMessage = (RTDS_MessageHeader *)eventParam1;
			
			if (!(timerUniqueId && event == RTDS_messageReceived))
				{
				/* Case of regular message event: add [-i mId] [-d msgData] options */
				#if defined(RTDS_SIMULATOR)
					mId = (unsigned long)(currentMessage->messageUniqueId);
				#endif
				msgData=(unsigned char *)(currentMessage->pData);

				if (msgData != NULL && RTDS_globalEventMessageDataEnable && event != RTDS_messageSaved)
					{

					RTDS_messageDataToString(&structuredData, currentMessage->messageNumber, currentMessage->dataLength, (void *)msgData, RTDS_PARAM_CODEC_MAX_DEPTH);

					extendedCommandString = (char *)RTDS_MALLOC(strlen(structuredData) + regularEventTraceSize);
					for (i=0; commandString[i] != '\0'; i++)
						extendedCommandString[i] = commandString[i];
					extendedCommandString[i] = '\0';

					RTDS_FREE(commandString);
					commandString = extendedCommandString;

					strcat(commandString, "-d| ");
					strcat(commandString, structuredData);
					strcat(commandString, "| ");
					RTDS_FREE(structuredData);

					#if defined(RTDS_SIMULATOR)
						sprintf(regularParameter,"-i| %lu| ", mId);
						strcat(commandString, regularParameter);
					#endif
					}
				else
					{
					#if defined(RTDS_SIMULATOR)
						/* If no message data only add mId */
						sprintf(regularParameter,"-i| %lu| ", mId);
						strcat(commandString, regularParameter);
					#endif
					}

				/* Set msgName sigNum parameter */
				sigNum = currentMessage->messageNumber;
				msgName = sigNum;
				}
			else
				{
				/* for timer */
				if (timerUniqueId)
					sigNum = currentMessage->messageNumber;
				}
			ignoreCurrentCommand = 0;
			break;
		case RTDS_semaphoreCreated:  /* command : "semaphoreCreated [-t time] [-N creatorName] [-B creatorNumber] 	 [-s semName] [-c creatorId] [-a stillAvailable]		 semId"  eventParam1 = semaphoreId and eventParam2 = NULL */

		case RTDS_semaphoreDeleted:  /* command : "semaphoreDeleted [-t time] [-N destructorName][-s semName] [-c destructorId] 										 semId"  eventParam1 = semaphoreId and eventParam2 = NULL */
			/* Add pId as optionnal parameter for RTDS_semaphoreCreated and RTDS_semaphoreDeleted event */
			if (pId != 0)
				{
				sprintf(regularParameter,"-c| 0x%lx| ", pId);
				strcat(commandString, regularParameter);
				}
		case RTDS_semTakeAttempt :	 /* command : "takeAttempt	 		[-t time] [-n pName] [-b pNumber] [-s semName] [-T timeout]							 pId semId"  eventParam1 = semaphoreId and eventParam2 = TIME_OUT */
		case RTDS_semTakeSucceded:	 /* command : "takeSucceeded		[-t time] [-n pName] [-b pNumber] [-s semName] [-a stillAvailable] pId semId"  eventParam1 = semaphoreId and eventParam2 = NULL*/
		case RTDS_semGive:					     /* command : "giveSem		   			[-t time] [-n pName] [-b pNumber] [-s semName] 																	   pId semId"  eventParam1 = semaphoreId and eventParam2 = NULL*/
		case RTDS_semTakeTimedOut:	 /* command : "takeTimedOut 		[-t time] [-n pName] [-b pNumber] [-s semName] 																	   pId semId"  eventParam1 = semaphoreId and eventParam2 = NULL*/
			/* Test if semaphore trace is activated */
			if (!RTDS_globalEventSemaphoreEnable) break;
			/* Set semaphore Id */
			semaphoreId = (unsigned long)eventParam1;

			/* Add semaphore name option */
			for (semInfo = RTDS_globalSemaphoreInfo; semInfo != NULL; semInfo = semInfo->next)
				if ((unsigned long)semInfo->semaphoreId == semaphoreId) break;
			if(semInfo != NULL)
				semNumber = semInfo->semaphoreNumber;
			if(semNumber != 0)
				{
				sprintf(regularParameter, "-S| %d| ", semNumber);
				strcat(commandString, regularParameter);
				}

			/* Add stillAvailable information if required */
			if (event == RTDS_semaphoreCreated || event == RTDS_semTakeSucceded)
				{
				/* Test semaphore to fill availibility information */
				stillAvailable = (long)eventParam2;
				sprintf(regularParameter, "-a| %ld| ", stillAvailable);
				strcat(commandString, regularParameter);
				}

			/* Add timeout information if required */
			if (event == RTDS_semTakeAttempt)
				{
				if ((long)eventParam2 == -1)
					sprintf(regularParameter, "-T| -1| ");
				else
					sprintf(regularParameter, "-T| %lu| ", (unsigned long)eventParam2);
				strcat(commandString, regularParameter);
				}

			ignoreCurrentCommand = 0;
			break;

		case RTDS_initDone: 				/* command = "initDone -t time" */
			ignoreCurrentCommand = 0;
			break;

		case	RTDS_timerStarted: /*  command	: "timerStarted [-t time] [-n pName] [-b pNumber] [-T timerName] pId tId [timeLeft]" eventParam1 = newTimerState and	eventParam2 = delay  */
			sigNum = (unsigned long)(((RTDS_TimerState *)eventParam1)->timerNumber);
			/* Set timeoutValue parameters */
			if(eventParam2 != (long)NULL)
				timeoutValue = (unsigned long)(eventParam2);
			else
				timeoutValue = (unsigned long)(((RTDS_TimerState *)eventParam1)->timeoutValue);
		case	RTDS_timerCancelled: /* command : "timerCancelled [-t time] [-n pName] [-b pNumber] [-T timerName] pId tId" 		eventParam1 = RTDS_timer and eventParam2 = NULL */
			/* Set timeoutValue parameters */
			sigNum = (unsigned long)(((RTDS_TimerState *)eventParam1)->timerNumber);
			ignoreCurrentCommand = 0;
			break;
		/* information message */
		case RTDS_information: 				/* command = "information -i bla bla bla" */
			ignoreCurrentCommand = 0;
			break;
		/* RTDS_systemError */
		case	RTDS_systemError:  /* command [-t time] [-n processName] processId errorId (errorId and processId inhex)] eventParam1 = errorNumber */
			errorId = eventParam1;
			ignoreCurrentCommand = 0;
			break;

		}

	/* If current message shouln't be ignored : add command parameters*/
	if(!ignoreCurrentCommand)
		{
	/* ADD GENERIC PARAMETER*/
		/* Add pId parameter for all valid command beside */
		if (event != RTDS_semaphoreCreated && event != RTDS_semaphoreDeleted)
			{
			sprintf(regularParameter,"0x%lx| ", pId);
			strcat(commandString, regularParameter);
			}

	/* ADD SPECIFIC PARAMETER*/
		if(event == RTDS_messageSaved || event == RTDS_messageSent || (event == RTDS_messageReceived && !timerUniqueId))
			{
			/* Add message event parameter : msgName and sigNum */
			sprintf(regularParameter,"%lu| %lu", msgName, sigNum);
			strcat(commandString, regularParameter);

			}
		else if (event == RTDS_semaphoreCreated || event == RTDS_semaphoreDeleted || event == RTDS_semTakeAttempt  || event == RTDS_semTakeSucceded || event == RTDS_semGive || event == RTDS_semTakeTimedOut)
			{
			/* Add semaphore Id parameter*/
			sprintf(regularParameter,"0x%lx| ", semaphoreId);
			strcat(commandString, regularParameter);
			}
		else if ((event == RTDS_messageReceived && timerUniqueId) || event == RTDS_timerStarted ||	event == RTDS_timerCancelled)
			{
			/* Add timer event parameter : tId parameter */
			sprintf(regularParameter,"%lu| ", sigNum);
			strcat(commandString, regularParameter);
			if (event == RTDS_timerStarted)
				{
				/* Add timeoutValue optionnal parameter if RTDS_timerStarted event */
				sprintf(regularParameter,"%lu| ", timeoutValue);
				strcat(commandString, regularParameter);
				}
			}
		else if (event == RTDS_sdlStateSet)
			{
			/* Add the state Name parameter */
			sprintf(regularParameter,"%ld| ", stateName);
			strcat(commandString, regularParameter);
			}
		else if (event == RTDS_information)
			{
			/* Add the information string */
			sprintf(regularParameter,"%s| ", (char *)eventParam1);
			strcat(commandString, regularParameter);
			}
		else if (event == RTDS_systemError)
			{
			sprintf(regularParameter, "0x%p| ", errorId);
			strcat(commandString,regularParameter);
			}

			/****************************/
			/*	Return formatted trace	*/
			/****************************/

		if (strlen(commandString) != 0)
			{
			/* Add end of command delimiter */
			strcat(commandString,"|\n");
			*formatedTrace = commandString;
			return RTDS_OK;
			}

		}
		/* should never be reached */
		return RTDS_ERROR;
	}

#endif /* RTDS_FORMAT_TRACE */


