/* PragmaDev RTDS OSE Delta integration */

#include "RTDS_MACRO.h"
#if defined(RTDS_SIMULATOR) || defined(RTDS_MSC_TRACE) || defined(RTDS_FORMAT_TRACE) || defined(RTDS_SOCKET_IP_ADDRESS)
#include "RTDS_Trace.h"
#endif

/* **************************************************************** *
 *	any_sig
 * **************************************************************** *
 * Specific to OSE to receive all signals when waiting for a
 * message
 * **************************************************************** */
static const SIGSELECT any_sig[] = {0};



/* **************************************************************** *
 *	RTDS_GetTimerUniqueId
 * **************************************************************** *
 * Get an available timer unique id.
 * Is used to set timers
 * **************************************************************** *
 * Parameters:
 *	 timerList: points to the first element of
 * the chained list of RTDS_TimerState
 * Returns:
 *	 The unique timer Id
 * **************************************************************** *
 * Context:
 *	 The chained list of RTDS_TimerState is sorted by increasing
 *	 unique timer id
 *	 timerUniqueId = NULL means the message is not a timer; so it
 *	 is avoided.
 * **************************************************************** */

long RTDS_GetTimerUniqueId(RTDS_TimerState *timerList)
	{
	RTDS_TimerState *RTDS_prevTimer,*RTDS_timer;
	long newTimerId;

	RTDS_CRITICAL_SECTION_START;
	/* If list is empty take 1 */
	if ( timerList == NULL )
		newTimerId = 1;
	/* If 1 is available, take it */
	else if ( timerList->timerUniqueId != 1 )
		newTimerId = 1;
	else
		{
		/* If there is a gap in the timerId chained list, take an Id in the gap */
		RTDS_prevTimer = timerList;
		newTimerId = 0;
		for ( RTDS_timer = timerList->next ; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
			{
			if ( RTDS_timer->timerUniqueId != RTDS_prevTimer->timerUniqueId + 1 ) newTimerId = RTDS_prevTimer->timerUniqueId + 1;
			RTDS_prevTimer = RTDS_timer;
			}
		/* No gap, let's take the next value */
		if ( newTimerId == 0 ) newTimerId = RTDS_prevTimer->timerUniqueId + 1;
		}

	RTDS_CRITICAL_SECTION_STOP;
	/* Check the counter did not go back to 0 */
	if ( newTimerId == 0 )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_NO_MORE_TIMER_UNIQUE_ID);
	return newTimerId;
	}



/* **************************************************************** *
 *	RTDS_StartTimer
 * **************************************************************** *
 * Starts a watchdog with the necessary parameters to create the
 * timer when it goes off.
 * **************************************************************** *
 * Parameters:
 *		 - QueueId of the receiver
 *		 - TimerNumber
 *		 - TimerUniqueId
 *		 - delay of the timer
 *		 - timerStateList: Address of the list of timer
 *		 - currentContext (RTDS_GlobalProcessInfo *): active task context
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_StartTimer(RTDS_QueueId queueId,
	long timerNumber,
	long timerUniqueId,
	int delay,
	RTDS_TimerState **pTimerStateList,
	RTDS_GlobalProcessInfo *RTDS_currentContext)
	{
	CANCEL_INFO 			watchDogId;
	RTDS_TimerState 	*timerState,*previousTimerState,*newTimerState;
	RTDS_MessageHeader *sig;


	/* Create a timer message */
	sig = (RTDS_MessageHeader *)alloc(sizeof(RTDS_MessageHeader), timerNumber);
	sig->messageNumber = timerNumber;
#if defined(RTDS_SIMULATOR) || defined(RTDS_MSC_TRACE) || defined(RTDS_FORMAT_TRACE)
	sig->messageUniqueId = 0;
#endif
	sig->timerUniqueId = timerUniqueId;
	sig->sender = queueId;
	sig->dataLength = 0;
	sig->pData = NULL;
	sig->next = NULL;

	requestTmo(&watchDogId, delay,(union SIGNAL **)(&sig));

	/* Update the list of timers */
	/* Important note: it is a sorted list based on the timerUniqueId field */
	newTimerState = (RTDS_TimerState *)RTDS_MALLOC(sizeof(RTDS_TimerState));
	if (newTimerState == NULL)
	RTDS_SYSTEM_ERROR(RTDS_ERROR_MALLOC_TIMER_STATE_IN_START_TIMER);

	/* Initialize the new element */
	newTimerState->state		= RTDS_TIMER_OK;
	newTimerState->timerNumber	= timerNumber;
	newTimerState->timeoutValue = delay+ get_ticks();
	newTimerState->timerUniqueId	= timerUniqueId;
	newTimerState->next 	= NULL;
	newTimerState->watchDogId 	= watchDogId;
	newTimerState->receiverQueueId = queueId;

	/* Insert the new element */
	if (*pTimerStateList == NULL) /* The list is empty */
	*pTimerStateList = newTimerState;
	else	/* The list is not empty */
	{
	previousTimerState = NULL;
	for (timerState = *pTimerStateList ; timerState != NULL ; timerState = timerState->next)
		{
		if (timerState->timerUniqueId > timerUniqueId)
		{
		if (previousTimerState == NULL)
			{
			*pTimerStateList = newTimerState;
			newTimerState->next = timerState;
			}
		else
			{
			previousTimerState->next = newTimerState;
			newTimerState->next = timerState;
			}
		break;
		}
		previousTimerState = timerState;
		}
	if (timerState == NULL) /* Inserted at the end of the list */
		previousTimerState->next = newTimerState;
	}

	RTDS_SIMULATOR_TRACE(RTDS_timerStarted, newTimerState, delay, RTDS_currentContext);

}



/* **************************************************************** *
 *	RTDS_StopTimer
 * **************************************************************** *
 * Stops a timer in trying to delete the watchdog. If unsuccessfull
 * set it cancelled in the timer chained list verified by the RTDS
 * kernel
 * **************************************************************** *
 * Parameters:
 *		 - TimerNumber
 *		 - pTimerStateList pointing to the timer chained list
 *		 - RTDS_currentContext currentContext of the task
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_StopTimer(
	long timerNumber,
	RTDS_TimerState **pTimerStateList,
	RTDS_GlobalProcessInfo *RTDS_currentContext )
	{
	RTDS_TimerState 	 *RTDS_timer, *RTDS_prevTimer;
	union SIGNAL			 *timerSignal;
	RTDS_MessageHeader *RTDS_message, *RTDS_prevMessage;


	RTDS_prevTimer = NULL;
	for ( RTDS_timer = *pTimerStateList ; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
		{
		if (( RTDS_timer->timerNumber == timerNumber ) && ( RTDS_timer->state != RTDS_TIMER_CANCELLED ))
			{
			RTDS_SIMULATOR_TRACE(RTDS_timerCancelled, RTDS_timer, NULL, RTDS_currentContext);

			/* Let's try to cancel the watchdog */
			timerSignal = cancelTmo(&(RTDS_timer->watchDogId));
			if ( timerSignal != 0 )
				{
				/* Anyway Remove it from the list */
				/* Is the first of the list */
				if (RTDS_prevTimer == NULL)
					{
					*pTimerStateList = RTDS_timer->next;
					}
				/* Not the first of the list */
				else
					{
					RTDS_prevTimer->next = RTDS_timer->next;
					}
				RTDS_FREE(RTDS_timer);

				/* Free the timer signal */
				free_buf(&timerSignal);
				}
			/* Could not cancel the timer. Probably went off allready */
			/* Set it cancelled in the list */
			else
				{
				RTDS_timer->state = RTDS_TIMER_CANCELLED;
				}
			return;
			}
		RTDS_prevTimer = RTDS_timer;
		}

	/* If execution gets here: the timer might be in the save queue */
	RTDS_prevMessage = NULL;
	for ( RTDS_message=RTDS_currentContext->readSaveQueue ; RTDS_message!=NULL ; RTDS_message=RTDS_message->next )
		{
		if ( RTDS_message->messageNumber == timerNumber )
			{
			/* Remove it from the list */
			/* Is the first of the list */
			if (RTDS_prevMessage == NULL)
				{
				RTDS_currentContext->readSaveQueue = RTDS_message->next;
				}
			/* Not the first of the list */
			else
				{
				RTDS_prevMessage->next = RTDS_message->next;
				}
			/* Free memory */
			free_buf( (union SIGNAL **)&(RTDS_message) );
			return;
			}
		RTDS_prevMessage = RTDS_message;
		}

	/* If execution gets here: the timer might be in the save queue */
	RTDS_prevMessage = NULL;
	for ( RTDS_message=RTDS_currentContext->writeSaveQueue ; RTDS_message!=NULL ; RTDS_message=RTDS_message->next )
		{
		if ( RTDS_message->messageNumber == timerNumber )
			{
			/* Remove it from the list */
			/* Is the first of the list */
			if (RTDS_prevMessage == NULL)
				{
				RTDS_currentContext->writeSaveQueue = RTDS_message->next;
				}
			/* Not the first of the list */
			else
				{
				RTDS_prevMessage->next = RTDS_message->next;
				}
			/* Free memory */
			free_buf( (union SIGNAL **)&(RTDS_message) );
			return;
			}
		RTDS_prevMessage = RTDS_message;
		}

	}



/* **************************************************************** *
 *	RTDS_GetProcessQueueId
 * **************************************************************** *
 * Returns the queueId of process based on its name
 * **************************************************************** *
 * Parameters:
 *		 - process name as a number defined in RTDS_gen.h
 * Returns:
 *	 the process queue id if found
 * Error:
 *	 System error call if not found and NULL returned
 * **************************************************************** */

RTDS_QueueId RTDS_GetProcessQueueId(int processNumber)
	{
	RTDS_GlobalProcessInfo	*processInfo;
	RTDS_QueueId						foundQueue = (RTDS_QueueId) NULL;

	RTDS_CRITICAL_SECTION_START;
	for (processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next)
		{
		if ( processInfo->processNumber == processNumber )
			{
			foundQueue = processInfo->myQueueId;
			break;
			}
		}
	RTDS_CRITICAL_SECTION_STOP;

	if (foundQueue == (RTDS_QueueId) NULL)
		RTDS_SYSTEM_ERROR(RTDS_ERROR_GET_PROCESS_QUEUE_ID);

	return foundQueue;
	}



/* **************************************************************** *
 *	RTDS_MsgQueueSend
 * **************************************************************** *
 * Send a message in a process's queue
 * **************************************************************** *
 * Parameters:
 *		 - messageNumber representing a message name
 *		 - dataLength length of data pointed by pData
 *		 - pData	pointer on data sent with the message
 *		 - receiver message receiver queue address
 *		 - sender 	message sender queue address
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_MsgQueueSend(
	long										messageNumber,
	long										dataLength,
	unsigned char 					*pData,
	RTDS_QueueId						receiver,
	RTDS_QueueId						sender,
	RTDS_GlobalProcessInfo	*RTDS_currentContext )
	{
	RTDS_MessageHeader *RTDS_messageToSend;

	RTDS_messageToSend = (RTDS_MessageHeader *)alloc(sizeof(RTDS_MessageHeader),messageNumber);

	RTDS_messageToSend->messageNumber = messageNumber;
	RTDS_messageToSend->timerUniqueId = 0;
	RTDS_messageToSend->sender = sender;
	RTDS_messageToSend->dataLength = dataLength;
	RTDS_messageToSend->pData = pData;
	RTDS_messageToSend->next = NULL;
	#if defined(RTDS_SIMULATOR) || defined(RTDS_MSC_TRACE) || defined(RTDS_FORMAT_TRACE)
		RTDS_messageToSend->messageUniqueId = RTDS_GetMessageUniqueId();
		RTDS_messageDataToString(&RTDS_globalPrintableParameters, messageNumber, dataLength, (void *)pData, RTDS_PARAM_CODEC_MAX_DEPTH);
		RTDS_SIMULATOR_TRACE(RTDS_messageSent, RTDS_messageToSend, receiver, RTDS_currentContext);
		RTDS_FREE(RTDS_globalPrintableParameters);
		RTDS_globalPrintableParameters = NULL;
	#endif

	send((union SIGNAL **)(&RTDS_messageToSend),receiver);
	}



/* **************************************************************** *
 *	RTDS_ProcessCreate
 * **************************************************************** *
 * Create a new SDL process and add a processInfo struct in the
 * process info chained list
 * **************************************************************** *
 * Parameters:
 *		 - name of the process as a char string
 *		 - name of the process as a number
 *		 - address of the process function
 *		 - priority fo the process
 *		 - the address of the address of the msg queue to be created
 *		 - the address of the msg queue of the parent process
 *		 - RTDS_currentContext currentContext of the task
 * Returns:
 *	 Nothing but updates the OFFSPRING of the caller
 * **************************************************************** */

void RTDS_ProcessCreate(
	char										*processName,
	int 										processNumber,
	void										*functionAddress,
	int 										priority,
	RTDS_QueueId						*pOffspringQueueId,
	RTDS_QueueId						selfQueueId,
	short 									synchronization,
	RTDS_GlobalProcessInfo	*RTDS_currentContext )
	{
	RTDS_GlobalProcessInfo * processInfo, *newProcessInfo;

	/* Allocate and fill in a new processInfo structure */
	newProcessInfo = (RTDS_GlobalProcessInfo *)RTDS_MALLOC(sizeof(RTDS_GlobalProcessInfo));
	if ( newProcessInfo == NULL )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_PROCESS_CREATE_MALLOC);

	/* Sets the process name as a number */
	newProcessInfo->processNumber = processNumber;
	/* SDL initial state */
	newProcessInfo->sdlState = 0;

	/* Last structure of the chained list */
	newProcessInfo->next							= NULL;
	newProcessInfo->parentQueueId 		= selfQueueId;
	newProcessInfo->offspringQueueId	= (RTDS_QueueId)NULL;
	newProcessInfo->currentMessage		= NULL;
	newProcessInfo->timerList 				= NULL;
#if defined(RTDS_SIMULATOR) || defined(RTDS_MSC_TRACE) || defined(RTDS_FORMAT_TRACE)
	newProcessInfo->priority					= priority;
#endif

	/* Task creation */
	/* Please note error handling is implicit */
	newProcessInfo->processId = create_process (
		OS_PRI_PROC,
		processName,
		(OSENTRYPOINT *)functionAddress,
		RTDS_TASK_STACK_SIZE,
		(OSPRIORITY) priority,
		(OSTIME) 0,
		(PROCESS) 0,
		(struct OS_redir_entry *) NULL,
		(OSVECTOR) 0,
		(OSUSER) 0);

	/* The queue id can be considered to be the pid because the queue is implicit in OSE */
	newProcessInfo->myQueueId = newProcessInfo->processId;
	*pOffspringQueueId = newProcessInfo->myQueueId;

	/* Add the process information to the chained list pointed by the RTDS_globalProcessInfo global variable */
	RTDS_CRITICAL_SECTION_START;
	if ( RTDS_globalProcessInfo == NULL )
		RTDS_globalProcessInfo = newProcessInfo;
	else
		{
		/* Let's get to the end of the list */
		for (processInfo = RTDS_globalProcessInfo ; processInfo->next != NULL ; processInfo = processInfo->next) ;
		processInfo->next = newProcessInfo;
		}
	RTDS_CRITICAL_SECTION_STOP;

	RTDS_SIMULATOR_TRACE(RTDS_processCreated, newProcessInfo, NULL, RTDS_currentContext);

	/* The newly created task can now run: RTDS_globalProcessInfo and trace are up to date. */
	if (synchronization == RTDS_NO_HOLD )
		start(newProcessInfo->processId);
	}



/* **************************************************************** *
 *	RTDS_ProcessKill
 * **************************************************************** *
 * Kills an SDL process and delete its queue and process info block
 * **************************************************************** *
 * Parameters:
 *	- the address of the process current context
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_ProcessKill( RTDS_GlobalProcessInfo *RTDS_currentContext )
	{
	RTDS_GlobalProcessInfo	*processInfo, *previousProcessInfo;
	RTDS_TimerState 	*RTDS_timer;
	RTDS_MessageHeader	*message, *tmpMessage;
	int 		pidToKill=0;

	/* Let's free the current message if any */
	if (RTDS_currentContext->currentMessage != NULL )
		{
		#if defined(RTDS_SIMULATOR) || defined(RTDS_MSC_TRACE) || defined(RTDS_FORMAT_TRACE)
			/* Release the message unique id back to the pool */
			RTDS_ReleaseMessageUniqueId(RTDS_currentContext->currentMessage->messageUniqueId);
		#endif

		/* Free memory */
		free_buf((union SIGNAL **)&(RTDS_currentContext->currentMessage));
		}


	/* Let's clean the timer chained list to free memory and stop watchdogs */
	for ( RTDS_timer = RTDS_currentContext->timerList ; RTDS_timer != NULL ; RTDS_timer = RTDS_currentContext->timerList)
	{
	RTDS_currentContext->timerList = RTDS_timer->next;
	RTDS_FREE(RTDS_timer);
	}

	/* Clean the save queue: free messages and message unique ids in the read and write save queues */
	for (message=RTDS_currentContext->readSaveQueue;message!=NULL;)
	{
	#if defined(RTDS_SIMULATOR) || defined(RTDS_MSC_TRACE) || defined(RTDS_FORMAT_TRACE)
		RTDS_ReleaseMessageUniqueId(message->messageUniqueId);
	#endif
	tmpMessage = message;
	message = message->next;
	free_buf((union SIGNAL **)&(tmpMessage));
	}

	for (message=RTDS_currentContext->writeSaveQueue;message!=NULL;)
	{
	#if defined(RTDS_SIMULATOR) || defined(RTDS_MSC_TRACE) || defined(RTDS_FORMAT_TRACE)
		RTDS_ReleaseMessageUniqueId(message->messageUniqueId);
	#endif
	tmpMessage = message;
	message = message->next;
	free_buf((union SIGNAL **)&(tmpMessage));
	}

	previousProcessInfo = NULL;
	RTDS_CRITICAL_SECTION_START;
	for (processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next)
	{
	/* The queue id is used to find the process id */
	if ( processInfo->myQueueId == RTDS_currentContext->myQueueId )  /* Found ! */
		{
		RTDS_SIMULATOR_TRACE(RTDS_processDied, processInfo, NULL, RTDS_currentContext);

		/* Update the process information chained list */
		if ( previousProcessInfo == NULL )	/* First one in the list */
			RTDS_globalProcessInfo = processInfo->next;
		else
			previousProcessInfo->next = processInfo->next;

		pidToKill = processInfo->processId;
		/* Delete the message queue and free the process info block */
		RTDS_FREE(processInfo);
		break;
		}
	previousProcessInfo = processInfo;
	}

	RTDS_CRITICAL_SECTION_STOP;

	/* Delete the task */
	if ( pidToKill != 0)
		{
		kill_proc(pidToKill);
		}
	else
		RTDS_SYSTEM_ERROR(RTDS_ERROR_TASK_TO_DELETE_NOT_FOUND);

	}





/* **************************************************************** *
 *	RTDS_Sem_Info_Insert
 * **************************************************************** *
 * Inserts a semaphoreInfo struct in the semaphore info chained list
 * **************************************************************** *
 * Parameters:
 *	 - semaphore number (name)
 *	 - id of the semaphore
 *	 - RTDS_currentContext currentContext of the task
 * Returns:
 *	 The semaphore id so that the user can use the semaphore id directly
 * **************************************************************** */

RTDS_SemaphoreId RTDS_Sem_Info_Insert(
		int 										semaphoreNumber,
		RTDS_SemaphoreId				semaphoreId,
		RTDS_GlobalProcessInfo	*RTDS_currentContext)
	{
	RTDS_GlobalSemaphoreInfo	*semInfo, *newSemInfo;

	/* Allocate and fill in a new semInfo structure */
	newSemInfo = (RTDS_GlobalSemaphoreInfo *)RTDS_MALLOC(sizeof(RTDS_GlobalSemaphoreInfo));
	if ( newSemInfo == NULL )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_SEM_INFO_INSERT_MALLOC);

	newSemInfo->semaphoreNumber = semaphoreNumber;
	newSemInfo->next						= NULL;

	/* Semaphore creation */
	newSemInfo->semaphoreId = semaphoreId;
	if ( newSemInfo->semaphoreId == NULL )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_SEM_INFO_INSERT);

	/* Add the semaphore information to the chained list pointed by the RTDS_globalSemaphoreInfo global variable */
	RTDS_CRITICAL_SECTION_START;
	if ( RTDS_globalSemaphoreInfo == NULL )
		RTDS_globalSemaphoreInfo = newSemInfo;
	else
		{
		/* Let's get to the end of the list */
		for (semInfo = RTDS_globalSemaphoreInfo ; semInfo->next != NULL ; semInfo = semInfo->next) ;
		semInfo->next = newSemInfo;
		}
	RTDS_CRITICAL_SECTION_STOP;

	RTDS_SIMULATOR_TRACE(RTDS_semaphoreCreated, semaphoreId, -1, RTDS_currentContext);

	return semaphoreId;
	}





/* **************************************************************** *
 *	RTDS_Sem_Delete
 * **************************************************************** *
 * Kills an RTOS semaphore and delete its info from the semaphore
 * information chained list and free the related memory
 * **************************************************************** *
 * Parameters:
 *		 - the address of the semaphore to find the semaphore info
 *		 block
 *		 - RTDS_currentContext currentContext of the task
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_Sem_Delete( RTDS_SemaphoreId semaphoreId, RTDS_GlobalProcessInfo	*RTDS_currentContext)
	{
	RTDS_GlobalSemaphoreInfo *semInfo, *semInfoPrev;

	RTDS_SIMULATOR_TRACE(RTDS_semaphoreDeleted, semaphoreId, 0, RTDS_currentContext);

	/* Remove the semaphore information from the chained list */
	semInfoPrev = NULL;
	RTDS_CRITICAL_SECTION_START;
	for (semInfo = RTDS_globalSemaphoreInfo ; semInfo != NULL ; semInfo = semInfo->next)
	{
	if ( semInfo->semaphoreId == semaphoreId) /* semaphore found */
		{
		if (semInfoPrev == NULL)
		RTDS_globalSemaphoreInfo = semInfo->next;
		else
		semInfoPrev->next = semInfo->next;

		/* The semaphore should be free for that operation */
		kill_sem(semInfo->semaphoreId);
		RTDS_FREE(semInfo);
		break;
		}
	semInfoPrev = semInfo;
	}
	RTDS_CRITICAL_SECTION_STOP;

	}




/* **************************************************************** *
 *	RTDS_SemaphoreIdTake
 * **************************************************************** *
 * Attempt to take a semaphore from its id
 * **************************************************************** *
 * Parameters:
 *	 - the id of the semaphore
 *	 - timeout value
 *	 - RTDS_currentContext currentContext of the task
 * Returns:
 *	 - status
 * **************************************************************** */

RTDS_SemaphoreStatus RTDS_SemaphoreIdTake(
	RTDS_SemaphoreId				semaphoreId,
	RTDS_SemaphoreTimeout 	timeOut,
	RTDS_GlobalProcessInfo	*RTDS_currentContext )
	{

	RTDS_SIMULATOR_TRACE(RTDS_semTakeAttempt, semaphoreId, timeOut, RTDS_currentContext);

	/* wait_sem allways blocking. No time out option possible. */
	/* System error if not the case */
	if (timeOut != RTDS_SEMAPHORE_TIME_OUT_FOREVER)
		RTDS_SYSTEM_ERROR(RTDS_ERROR_SEMAPHORE_TIME_OUT);

	wait_sem(semaphoreId);

	RTDS_SIMULATOR_TRACE(RTDS_semTakeSucceded, semaphoreId, (long)get_sem(semaphoreId), RTDS_currentContext);

	return RTDS_OK;
	}




/* **************************************************************** *
 *	RTDS_GetSemaphoreId
 * **************************************************************** *
 * Gets the id of a semaphore from its number (name)
 * **************************************************************** *
 * Parameters:
 *	 - semaphore number representing its name
 * Returns:
 *	 - the id of the semaphore
 * **************************************************************** */

RTDS_SemaphoreId RTDS_GetSemaphoreId( int semaphoreNumber )
	{
	RTDS_GlobalSemaphoreInfo	*semInfo;
	RTDS_SemaphoreId			foundSemaphoreId;

	foundSemaphoreId = NULL;
	RTDS_CRITICAL_SECTION_START;
	for (semInfo = RTDS_globalSemaphoreInfo ; semInfo != NULL ; semInfo = semInfo->next)
		{
		if ( semInfo->semaphoreNumber == semaphoreNumber ) /* semaphore found */
			{
			foundSemaphoreId = semInfo->semaphoreId;
			break;
			}
		}
	RTDS_CRITICAL_SECTION_STOP;

	if (foundSemaphoreId==NULL)
		RTDS_SYSTEM_ERROR(RTDS_ERROR_GET_SEMAPHORE_ID);

	return foundSemaphoreId;	/* Might cause an RTOS exception if NULL*/
	}



/* **************************************************************** *
 *	RTDS_TransitionCleanUp
 * **************************************************************** *
 * Called at the end of transitions:
 * - frees message buffer if valid
 * - re-organize save queue if state has changed
 * **************************************************************** *
 * Parameters:
 *	 None
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_TransitionCleanUp(RTDS_GlobalProcessInfo *RTDS_currentContext, int RTDS_sdlStatePrev)
	{
	/* Free message buffer if valid */
	if (RTDS_currentContext->currentMessage != NULL )
		{
    /* Free the message parameter structure if any */
    if (RTDS_currentContext->currentMessage->pData != NULL){
      RTDS_FREE(RTDS_currentContext->currentMessage->pData);
      }
		#ifdef RTDS_SIMULATOR
			/* Release the message unique id back to the pool */
			RTDS_ReleaseMessageUniqueId(RTDS_currentContext->currentMessage->messageUniqueId);
		#endif

		/* Free memory */
		free_buf((union SIGNAL **)&(RTDS_currentContext->currentMessage));
		RTDS_currentContext->currentMessage = NULL;
		} /* End of else if ( RTDS_currentContext->currentMessage != NULL ) */

	/* If SDL state has changed and messages have been saved: reorganise the save queue */
	if ( (RTDS_currentContext->sdlState != RTDS_sdlStatePrev) && (RTDS_currentContext->writeSaveQueue != NULL) )
		{
		RTDS_MessageHeader	*message;
		/* Let's get to the end of the save queue */
		for (message=RTDS_currentContext->writeSaveQueue ; message->next != NULL ; message = message->next);
		message->next = RTDS_currentContext->readSaveQueue;
		RTDS_currentContext->readSaveQueue = RTDS_currentContext->writeSaveQueue;
		RTDS_currentContext->writeSaveQueue = NULL;
		}

	return;
	}



/* **************************************************************** *
 *	RTDS_GetSystemTime
 * **************************************************************** *
 * As its name states...
 * **************************************************************** *
 * Parameters:
 *	 None
 * Returns:
 *	 System tick count value
 * **************************************************************** */

unsigned long RTDS_GetSystemTime(void)
	{
	return get_systime(NULL);
	}




/* **************************************************************** *
 * **************************************************************** *
 * **************************************************************** *
 * THE CODE BELOW IS ONLY USED TO DEBUG WITH RTDS SDL-RT DEBUGGER
 * **************************************************************** *
 * **************************************************************** *
 * **************************************************************** */
#if defined(RTDS_SIMULATOR) || defined(RTDS_MSC_TRACER) || defined(RTDS_FORMAT_TRACE)
unsigned char 					*RTDS_globalMessageUniqueIdPool=NULL;
RTDS_GlobalTraceInfo		RTDS_globalTraceEntry={RTDS_systemError,NULL,0};
char										*RTDS_globalPrintableParameters=NULL;
OSTICK									RTDS_globalSystemTime;




/* **************************************************************** *
 *	RTDS_GetMessageUniqueId
 * **************************************************************** *
 * Gets a message unique id for the simulator
 * **************************************************************** *
 * Parameters:
 *	 None
 * Returns:
 *	 - the message unique id (minimum is 1)
 * **************************************************************** */

unsigned long RTDS_GetMessageUniqueId(void)
	{
	unsigned char *index;
	long		uniqueByteId;
	long		uniqueBitId;

	index=RTDS_globalMessageUniqueIdPool;



	RTDS_CRITICAL_SECTION_START;
	for (uniqueByteId=0;uniqueByteId < RTDS_MESSAGE_UNIQUE_ID_POOL_SIZE;uniqueByteId++)
	{
	if (*index != 0xFF)
		{
		for (uniqueBitId=0;uniqueBitId<8;uniqueBitId++)
		{
		if ( ( (1<<uniqueBitId) & *index) ==0 )
			{
			*index = *index | (1<<uniqueBitId) ;
			RTDS_CRITICAL_SECTION_STOP;
			return 8*uniqueByteId+uniqueBitId+1;
			}
		}
		}
	index++;
	}
	RTDS_CRITICAL_SECTION_STOP;

	/* All bits are set... No more message unique id */
	RTDS_SYSTEM_ERROR(RTDS_ERROR_NO_MORE_MSG_UNIQUE_ID);
	return 0;
	}




/* **************************************************************** *
 *	RTDS_ReleaseMessageUniqueId
 * **************************************************************** *
 * Make a message unique id available from the pool
 * **************************************************************** *
 * Parameters:
 *	 message unique id
 * Returns:
 *	 nothing
 * **************************************************************** */

void RTDS_ReleaseMessageUniqueId(unsigned long messageUniqueId)
	{
	unsigned char *index;

	if (messageUniqueId == 0) /* probably a timer */
		return;

	messageUniqueId -= 1;
	index = RTDS_globalMessageUniqueIdPool;
	index += (unsigned char)(messageUniqueId/8);
	RTDS_CRITICAL_SECTION_START;
	(*index) = (*index) & ~(1<<messageUniqueId%8);
	RTDS_CRITICAL_SECTION_STOP;
	}


/* **************************************************************** *
 *	RTDS_SimulatorMsgQueueSend
 * **************************************************************** *
 * Send a message in a process's queue without tracing
 * **************************************************************** *
 * Parameters:
 *		 - messageNumber representing a message name
 *		 - dataLength length of data pointed by pData
 *		 - pData	pointer on data sent with the message
 *		 - receiver message receiver queue address
 *		 - sender 	message sender queue address
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_SimulatorMsgQueueSend(
	long										messageNumber,
	long										dataLength,
	unsigned char 					*pData,
	RTDS_QueueId						receiver,
	RTDS_QueueId						sender,
	RTDS_GlobalProcessInfo	*RTDS_currentContext )
	{
	RTDS_MessageHeader *RTDS_messageToSend;

	RTDS_messageToSend = (RTDS_MessageHeader *)alloc(sizeof(RTDS_MessageHeader),messageNumber);

	RTDS_messageToSend->messageNumber = messageNumber;
	RTDS_messageToSend->timerUniqueId = 0;
	RTDS_messageToSend->sender = sender;
	RTDS_messageToSend->dataLength = dataLength;
	RTDS_messageToSend->pData = pData;
	RTDS_messageToSend->next = NULL;

	RTDS_messageToSend->messageUniqueId = RTDS_GetMessageUniqueId();
	RTDS_SimulatorTrace((enum RTDS_EventType)RTDS_messageSent, (void *)RTDS_messageToSend, (long)receiver, (RTDS_GlobalProcessInfo *)RTDS_currentContext, RTDS_DTRACE_ACK_NOWAIT);

	send((union SIGNAL **)(&RTDS_messageToSend),receiver);
	}




/* **************************************************************** *
 *	RTDS_SimulatorTrace
 * **************************************************************** *
 * As its name states... Handles trace with the host depending on
 * the type of link available and options set in the profile.
 * **************************************************************** *
 * Parameters:
 *	enum RTDS_EventType  event: 						trace event
 *	void *eventParameter1:									parameter 1 of the trace
 *	long eventParameter2: 									parameter 2 of the trace
 *	RTDS_GlobalProcessInfo *currentContext: calling context
 *	int waitAck 														should the trace be acknowledged (1) or not (0)
 * Returns:
 *	 nothing
 * **************************************************************** */

void RTDS_SimulatorTrace(
		enum RTDS_EventType  event,
		void *eventParameter1,
		long eventParameter2,
		RTDS_GlobalProcessInfo *currentContext,
		int waitAck)
	{
	#ifdef RTDS_FORMAT_TRACE
		char *formatedTrace;
		int retValue;

		/* Do not take semaphore when no ack is required
		i.e. when it comes from the debugger
		i.e. executed in the context of the task reading the socket
		otherwise dead lock ! */
		if (waitAck == RTDS_DTRACE_ACK_WAIT)
			{RTDS_CRITICAL_TRACE_SECTION_START;}

		RTDS_globalTraceEntry.event = event;
		RTDS_globalTraceEntry.eventParameter1 = (void *) eventParameter1;
		RTDS_globalTraceEntry.eventParameter2 = (long) eventParameter2;
		RTDS_globalTraceEntry.currentContext = (RTDS_GlobalProcessInfo *)currentContext;

		#ifdef RTDS_SIMULATOR
			retValue = RTDS_FormatTrace(
			(unsigned long)RTDS_GetSystemTime(),	/* System time */
			RTDS_globalTraceEntry,								/* Trace info */
			waitAck,															/* Acknowledgment except when sending messages from the SDL-RT debugger*/
			&formatedTrace);											/* Buffer will be allocated by RTDS_FormatDynamicTrace */
		#else
			retValue = RTDS_FormatTrace(
			(unsigned long)RTDS_GetSystemTime(),	/* System time */
			RTDS_globalTraceEntry,								/* Trace info */
			RTDS_DTRACE_ACK_NOWAIT, 							/* No acknowledgment */
			&formatedTrace);											/* Buffer will be allocated by RTDS_FormatDynamicTrace */
		#endif

		if ( (retValue == RTDS_OK) && (formatedTrace!=NULL) )
			{
			#ifdef RTDS_SOCKET_IP_ADDRESS
				RTDS_SOCKET_ACCESS_TAKE
				RTDS_SendSocket(formatedTrace,strlen(formatedTrace));
				RTDS_SOCKET_ACCESS_GIVE
			#endif
			if (waitAck == RTDS_DTRACE_ACK_WAIT)
				{ RTDS_DTRACE_ACKNOWLEDGE_WAIT; }
			}

		RTDS_FREE(formatedTrace);

		RTDS_DummyTraceFunction();

		if (waitAck == RTDS_DTRACE_ACK_WAIT)
			{RTDS_CRITICAL_TRACE_SECTION_STOP;}
	#endif
	}






/* **************************************************************** *
 *	RTDS_DummyTraceFunction
 * **************************************************************** *
 * As its name states... The simulator sets a breakpoint on this
 * function and reads the RTDS_globalTraceEntry variable to see
 * what happened
 * **************************************************************** *
 * Parameters:
 *	 None
 * Returns:
 *	 nothing
 * **************************************************************** */

void RTDS_DummyTraceFunction(void)
	{
	}

#endif /* defined(RTDS_SIMULATOR) || defined(RTDS_MSC_TRACE) || defined(RTDS_FORMAT_TRACE) */

