/* **************************************************************** *
 *      PragmaDev RTDS CMX integration for Tasking compiler
 * **************************************************************** *
 * Tick count time handling &
 * Function to dynamically allocate and free semaphore ids
 * **************************************************************** */

#ifdef _C166
#include <reg167.h>
#endif
#ifdef __arm__
#include <lpc213x.h>
#endif
#include "RTDS_OS.h"
#include "RTDS_MACRO.h"
#include "RTDS_Cmx.h"

#ifdef _C166
/* *********************************************************************** *
 *	Module to set up T2 as a tick timer to be used for CMX RTOS timer.
 *	After initialisation time slicing and cyclic timers, etc is enabled.
 *	T2 uses software reload in the interrupt routine itself
 *	user may adapt TickTimerInit and the interrupt routine to for instance:
 *	- T2 reload mode (this is not done for clearity, and spares T3)
 *	- PEC channel service to reload the timer value
 *	- other user solutions
 * *********************************************************************** */

/* Timer 2 functioning setup	parameters	 */
#define T2UDEval	0 	/* use internal clock input 		*/
#define T2UDval 	1 	/* down counting			*/
#define T2Rval		1 	/* timer runs immediately 			*/
#define T2Mval		0 	/* timer mode 				*/
#define T2Ival		0 	/* prescaler 8				*/
#define T2CONval		(T2UDEval<<8) | (T2UDval<<7) | (T2Rval<<6) | (T2Mval<<3) | (T2Ival)

#define T2GLVLval 3 	/* group level 3				*/
#define T2ILVLval 11		/* int priority level 11		*/
#define T2IEval 	1
#define T2prval 	(T2IEval<<6) | (T2ILVLval<<2) | (T2GLVLval)  /* priority */

#define T2initval 	3000		/* counter reload value 		*/
	/* so interrupt is called every 8 * 3000 = 24000 clockcycles = every 1.2 ms at 20 MHz */


/* **************************************************************** *
 *	TickTimerInit
 * **************************************************************** *
 * Initialise timer with predefined values
 * To be called before K_OS_Start()
 * **************************************************************** *
 * Parameters:
 *		 - nothing
 * Returns:
 *	 - nothing
 * **************************************************************** */
void TickTimerInit(void)
{
	T2=T2initval; /* could also be done with a PEC channel interrupt! */
	T2IC=T2prval;
	T2CON = T2CONval; 	 /* set up timer as specified */
}


/* **************************************************************** *
 *	TickTimerint
 * **************************************************************** *
 * Timer 2 interrupt service routine example to call
 * K_OS_Tick_Update() do NOT USE #pragma noframe, because
 * Tick_Update needs proper stack save/restore!!
 * **************************************************************** *
 * Parameters:
 *		 - nothing
 * Returns:
 *	 - nothing
 * **************************************************************** */
interrupt (0x22) void TickTimerInt(void)
{
/* K_OS_Intrp_Entry();		not used for C166 */

	T2=T2initval;
	K_OS_Tick_Update(); 	/* do reschedule				*/
	K_OS_Intrp_Exit();		/* use exit 			*/

}
#endif

#ifdef __arm__
/* **************************************************************** *
 *	TickTimerInit
 * **************************************************************** *
 * Initialise timer with predefined values
 * To be called before K_OS_Start()
 * **************************************************************** *
 * Parameters:
 *		 - nothing
 * Returns:
 *	 - nothing
 * **************************************************************** */
void TickTimerInit(void)
{
	/*
	the following will set up the CMX timer tick.
	Assumes 60 Mhz clock.
	*/
   T0MR0 = 149999;                     /* 10mSec = 150.000-1 counts */
   T0MCR = 3;                          /* Interrupt and Reset on MR0 */
   T0TCR = 1;                          /* Timer0 Enable */
   VICVectAddr0 = (word32)cmx_irq; /* set interrupt vector in 0 */
   VICVectCntl0 = 0x20 | 4;            /* use it for Timer 0 Interrupt */
   VICIntEnable = 0x00000010;          /* Enable Timer0 Interrupt */
}

void cmx_undef(void)
{
	/* put code here */
}

void cmx_swi(int SWI_num)
{
	switch (SWI_num)
		{
		/* put code here */
		default:
			break;
		}
}

void cmx_prefetch(void)
{
	/* put code here */
}

void cmx_abort(void)
{
	/* put code here */
}


void cmx_irq(void)
{
   if (T0IR & 0x01)
      {
      if (TEST_CMX_ACTIVE)
         K_OS_Tick_Update();       /* call the CMX tick function */

      T0IR        = 1;             /* Clear interrupt flag */
      }

   VICVectAddr = 0;            /* Acknowledge Interrupt */
}

void cmx_fiq(void)
{
	/* put code here */
}

#endif


/* **************************************************************** *
 *	Function to dynamically allocate and free semaphore ids
 * **************************************************************** */

/* **************************************************************** *
 * RTDS_globalSemaphoreId is an array of short. If element n of the
 * array is 1 the semaphore n is not available, if it is 0 it is
 * available
 * **************************************************************** */
unsigned char RTDS_globalSemaphoreId[C_MAX_SEMAPHORES];


/* **************************************************************** *
 *	RTDS_initQueue
 * **************************************************************** *
 * Makes all mailboxes available.
 * **************************************************************** *
 * Parameters:
 *		 - nothing
 * Returns:
 *	 - nothing
 * **************************************************************** */
void RTDS_initResourceId(void)
	{
	int i;
	for (i=0;i<C_MAX_SEMAPHORES;i++)
		RTDS_globalSemaphoreId[i]=0;
	}


/* **************************************************************** *
 *	RTDS_semaphoreIdGet
 * **************************************************************** *
 * Since CMX has static semaphores up to C_MAX_SEMAPHORES. This
 * function returns an available semaphore identifier.
 * **************************************************************** *
 * Parameters:
 *		 - nothing
 * Returns:
 *	 - semahore identifier or calls RTDS_SYSTEM_ERROR
 * **************************************************************** */
int RTDS_semaphoreIdGet(void)
	{
	int i;
	for (i=0;(RTDS_globalSemaphoreId[i]!=0) && (i<C_MAX_SEMAPHORES);i++) ;
	if ( i!=C_MAX_SEMAPHORES )
		{
		RTDS_globalSemaphoreId[i]=1;
		return i;
		}
	else
		{
		RTDS_SYSTEM_ERROR(RTDS_ERROR_SEMAPHORE_GET);
		return -1;
		}
	}

/* **************************************************************** *
 *	RTDS_semaphoreIdFree
 * **************************************************************** *
 * Since CMX has static semaphores up to C_MAX_SEMAPHORES. This
 * function returns an available mailbox identifier.
 * **************************************************************** *
 * Parameters:
 *		 - nothing
 * Returns:
 *	 - nothing
 * **************************************************************** */
void RTDS_semaphoreIdFree(int semaphoreIndex)
	{
	RTDS_globalSemaphoreId[semaphoreIndex] = 0;
	}


