#!/bin/sh

# ========================================
# INSTALLATION SCRIPT FOR PRAGMADEV STUDIO
# ========================================
# Copyright (c) PragmaDev 2002~2020
#
# History:
# ------------------------------------------------------------
# Version | Comment
# ------------------------------------------------------------
# 1.0     | Creation
# 1.1     | Exclude share/3rdparty/MinGW when installing on
#         | and for Unix
# 1.2     | Clean-up target directory before installing
# 2.0     | Adaptation for PragmaDev Studio
# 2.1     | Added installation for desktop menu entry & file
#         | types.
# 2.2     | Replaced linux by linux64.
# 2.3     | Reverted back linux64 to linux + raspbian support
# 2.4     | Support for both linux & linux64
# 2.5     | Removed definition of LD_LIBRARY_PATH in
#         | configuration script since it caused issues on
#         | systems because of library version conflicts
# ------------------------------------------------------------

## Function reading a line from the terminal
getAnswer()
{
  ## Read from terminal
  read answer
  ## Quit if needed
  if [ "$answer" = "q" ]
  then
    exit 0
  fi
  ## Output answer
  echo $answer
}

## Welcome screen
cat <<.
=============================
PRAGMADEV STUDIO INSTALLATION
=============================
This is the installation script for PragmaDev Studio, PragmaDev Specifier,
PragmaDev Developer and PragmaDev Tester.

Please select the application(s) you wish to install:

  (1) PragmaDev Studio
  (2) PragmaDev Specifier
  (3) PragmaDev Developer
  (4) PragmaDev Tester
  (q) Abort installation and quit

Please enter one or several digits separated with a space (e.g '4' or '1 3')
or 'q' to quit.
.
application_numbers=`getAnswer`
install_studio="no"
install_specifier="no"
install_developer="no"
install_tester="no"
application_label=""
for i in $application_numbers
do
  case $i in
    1)
      install_studio="yes"
      if [ "$application_label" = "" ]
      then
        application_label="PragmaDev Studio"
      else
        application_label="PragmaDev applications"
      fi
      ;;
    2)
      install_specifier="yes"
      if [ "$application_label" = "" ]
      then
        application_label="PragmaDev Specifier"
      else
        application_label="PragmaDev applications"
      fi
      ;;
    3)
      install_developer="yes"
      if [ "$application_label" = "" ]
      then
        application_label="PragmaDev Developer"
      else
        application_label="PragmaDev applications"
      fi
      ;;
    4)
      install_tester="yes"
      if [ "$application_label" = "" ]
      then
        application_label="PragmaDev Tester"
      else
        application_label="PragmaDev applications"
      fi
      ;;
    *)
      echo "Unknown application number $i - ignored"
      ;;
  esac
done
if [ "$install_studio" = "no" -a "$install_specifier" = "no" -a "$install_developer" = "no" -a "$install_tester" = "no" ]
then
  echo "Nothing to install; aborting installation."
  exit 1
fi

## Architecture may be passed as an argument
arch=$1

## If it wasn't, try to figure it out
if [ "$arch" = "" ]
then
  case `uname -s` in
    Linux)
      if [ "`uname -m`" = "armv7l" ]
      then
        arch='raspbian'
      elif [ "`uname -m`" = "x86_64" ]
      then
        arch='linux64'
      else
        arch='linux'
      fi
      ;;
    SunOS)
      case `uname -r` in
        5.10)
          arch='solaris10intel'
          ;;
        *)
          arch='solaris8sparc'
          ;;
        esac
      ;;
  esac
fi

## If architecture still not found or invalid, can't proceed
if [ "$arch" != "linux" -a "$arch" != "linux64" -a "$arch" != "raspbian" -a "$arch" != "solaris8sparc" -a "$arch" != "solaris10intel" ]
then
  echo '!!! ERROR !!!'
  echo 'Unable to determine architecture.'
  echo 'You may try to re-run this installation script passing the architecture'
  echo 'as an argument. Valid architectures are "linux", "linux64", "raspbian",'
  echo '"solaris8sparc" and "solaris10intel".'
  exit 1
fi

## Directory for install script is the mount point of the CD-ROM
rootDir=`dirname $0`

## Display license agreement
more $rootDir/doc/LICENSE.txt
echo ""
echo "Answer 'yes' to accept this license agreement. Any other answer will quit the installation."
license_answer=`getAnswer`
if [ "$license_answer" != "yes" ]
then
  echo "Licence refused. Quitting..."
  exit 0
fi

## Ask for installation directory
installDir=""
while [ "$installDir" = "" ]
do
  echo ""
  echo "Please enter the installation directory for $application_label"
  echo "(default: /opt/pragmadev/pragmastudio; 'q' to quit)"
  installDir=`getAnswer`
  if [ "$installDir" = "" ]
  then
    installDir='/opt/pragmadev/pragmastudio'
  fi

  ## If installation directory already exists
  if [ -r "$installDir" ]
  then
    ## If it's not a directory, error...
    if [ ! -d "$installDir" ]
    then
      echo '!!! ERROR !!!'
      echo "File $installDir exists, but is not a directory. Please choose another"
      echo "location."
      echo ""
      echo "Press return to continue or 'q' to quit"
      getAnswer > /dev/null
      installDir=""
    else
      ## If directory is not writable, error
      if [ ! -w "$installDir" ]
      then
        echo '!!! ERROR !!!'
        echo "Directory $installDir exists, but is not writable. Please choose another"
        echo "location, or quit the installation and re-log as a user with sufficient"
        echo "rights to write in this directory"
        echo ""
        echo "Press return to continue or 'q' to quit"
        getAnswer > /dev/null
        installDir=""
      ## If directory is writable
      else
        ## Directory will be emptied; ask confirmation
        echo "--- WARNING! ---"
        echo "Existing directory $installDir will be emptied before installation!"
        echo "Do you really want to install in $installDir ([y]/n/q)?"
        answer=`getAnswer`
        if [ "$answer" = "n" ]
        then
          installDir=""
        fi
      fi
    fi
  ## If installation directory does not exist, ask if we should create it
  else
    echo "Directory $installDir does not exist. Do you want to create it ([y]/n/q)?"
    answer=`getAnswer`
    if [ "$answer" != "n" ]
    then
      mkdir -p $installDir
      ## If creation failed
      if [ $? -ne 0 ]
      then
        echo '!!! ERROR !!!'
        echo "Could not create installation directory $installDir"
        echo "Make sure you have the appropriate rights or choose another directory"
        echo ""
        echo "Press return to continue or enter 'q' to quit"
        getAnswer > /dev/null
        installDir=""
      fi
    else
      installDir=""
    fi
  fi
done

## Desktop menu, mimetypes & icon configuration
desktop_install=NO
if [ "`which xdg-desktop-menu 2> /dev/null`" = "" -o "`which xdg-mime 2> /dev/null`" = "" -o  "`which xdg-icon-resource 2> /dev/null`" = "" ]
then
  desktop_install=NO_XDG
else
  echo "The XDG utilities are installed on your system, so an entry to the desktop menu"
  echo "as well as the definitions for the file types for $application_label can be"
  echo "installed."
  if [ "`whoami`" = "root" ]
  then
    echo "Since you are root, the menu and file types will be installed system-wide."
    echo "Install menu entry & file types ([y]/n/q)?"
  else
    echo "Do you want to create a entry in the desktop menu and define the file types?"
    echo "  - n = no"
    echo "  - u = for this user only"
    echo "  - s = system-wide (requires root permissions)"
    echo "(n/[u]/s/q):"
  fi
  answer=`getAnswer`
  case $answer in
    u)
      desktop_install=user
      ;;
    y)
      desktop_install=system
      ;;
    s)
      desktop_install=sudo
      ;;
  esac
fi

## Confirmation screen
cat <<.

$application_label will now be installed with the following parameters:

 - Architecture: $arch
 
 - Installation directory: $installDir

PLEASE CHECK THE INFORMATIONS ABOVE. If you press return now, $application_label
will be installed. If any of the informations above is not correct, enter 'q'
and re-run the installation script from the beginning.
.
getAnswer > /dev/null

## Clean-up destination directory
echo '--- Cleaning up installation directory'
( cd $installDir ; rm -rf `ls -A` )

## Install binaries
echo "--- Installing $application_label binaries"
cd $rootDir
if [ ! -r $installDir/bin ]
then
  mkdir $installDir/bin
fi
cp -r bin/$arch/* $installDir/bin/
## Clean up unneeded wrapper scripts
if [ "$install_studio" = "no" ]
then
  rm $installDir/bin/pragmastudio
fi
if [ "$install_specifier" = "no" ]
then
  rm $installDir/bin/pragmaspecifier
fi
if [ "$install_developer" = "no" ]
then
  rm $installDir/bin/pragmadeveloper
fi
if [ "$install_tester" = "no" ]
then
  rm $installDir/bin/pragmatester
fi

## Install documentation
echo "--- Installing $application_label documentation"
tar cf - doc | (cd $installDir ; tar xf -)

## Install common files
## NB: the --exclude tar option is a GNU tar extension; don't try to use it on platforms
## that do not have GNU tar installed by default (namely Solaris)
echo "--- Installing $application_label common files"
if [ "$arch" = "linux" -o "$arch" = "linux64" -o "$arch" = "raspbian" ]
then
  tarOptions='--exclude=share/3rdparty/MinGW --exclude=share/3rdparty/MinGW_3.4.5 --exclude=share/3rdparty/demoddix/windows --exclude=share/3rdparty/Diversity/windows --exclude=share/3rdparty/ns-3.10/pragmadev/windows --exclude=share/3rdparty/Reqtify'
else
  tarOptions=''
fi
tar cf - $tarOptions share | (cd $installDir ; tar xf -)
## Change permissions on share/ccg to make sure files won't be deleted by mistake
chmod -R a-w $installDir/share/ccg

## Install contributed files
if [ -d contrib ]
then
  echo '--- Installing contributed files'
  tar cf - contrib | (cd $installDir ; tar xf -)
fi

## Install examples
echo "--- Installing $application_label example files"
tar cf - examples | (cd $installDir ; tar xf -)

## Creating configuration scripts
echo '--- Creating configuration scripts'
echo "RTDS_HOME=$installDir"                >  $installDir/pragmastudio-config.sh
echo "MSCT_HOME=$installDir"                >> $installDir/pragmastudio-config.sh
echo 'PATH="$PATH":$RTDS_HOME/bin'          >> $installDir/pragmastudio-config.sh
echo 'export RTDS_HOME MSCT_HOME PATH'      >> $installDir/pragmastudio-config.sh

echo "setenv RTDS_HOME $installDir"         >  $installDir/pragmastudio-config.csh
echo "setenv MSCT_HOME $installDir"         >> $installDir/pragmastudio-config.csh
echo 'setenv PATH "$PATH":$RTDS_HOME/bin'   >> $installDir/pragmastudio-config.csh

## Desktop menu & icons
if [ "$desktop_install" = "user" -o "$desktop_install" = "system" -o "$desktop_install" = "sudo" ]
then
  echo '--- Installing desktop menu & icons'
  pushd /var/tmp
  needs_sudo=NO
  if [ "$desktop_install" = "sudo" ]
  then
    needs_sudo=YES
    desktop_install=system
  fi
  ## Desktop application configuration file
  echo "[Desktop Entry]"                                                                                                >  pragmadev-PragmaDevStudio.desktop
  echo "Type=Application"                                                                                               >> pragmadev-PragmaDevStudio.desktop
  echo "Name=PragmaDev Studio"                                                                                          >> pragmadev-PragmaDevStudio.desktop
  echo "GenericName=Modeling and testing tool for real-time and embedded systems"                                       >> pragmadev-PragmaDevStudio.desktop
  echo "Icon=$RTDS_HOME/share/icons/common/PragmaDevStudio.svg"                                                         >> pragmadev-PragmaDevStudio.desktop
  echo "Exec=$RTDS_HOME/bin/pragmastudio "'"%f"'                                                                        >> pragmadev-PragmaDevStudio.desktop
  echo "Terminal=false"                                                                                                 >> pragmadev-PragmaDevStudio.desktop
  echo "MimeType=application/pragmastudioproject;application/pragmastudiodiagram;application/pragmastudioprotoguipkg;"  >> pragmadev-PragmaDevStudio.desktop
  echo "Categories=Development;IDE"                                                                                     >> pragmadev-PragmaDevStudio.desktop
  ## Mimetypes file
  echo '<?xml version="1.0" encoding="UTF-8"?>'                                             >  pragmadev-PragmaDevStudio.xml
  echo '<mime-info xmlns="http://www.freedesktop.org/standards/shared-mime-info">'          >> pragmadev-PragmaDevStudio.xml
  echo '  <mime-type type="application/pragmastudioproject">'                               >> pragmadev-PragmaDevStudio.xml
  echo '         <comment xml:lang="en">PragmaDev Studio project</comment>'                 >> pragmadev-PragmaDevStudio.xml
  echo '         <glob pattern="*.rdp"/>'                                                   >> pragmadev-PragmaDevStudio.xml
  echo '  </mime-type>'                                                                     >> pragmadev-PragmaDevStudio.xml
  echo '  <mime-type type="application/pragmastudiodiagram">'                               >> pragmadev-PragmaDevStudio.xml
  echo '         <comment xml:lang="en">PragmaDev Studio diagram</comment>'                 >> pragmadev-PragmaDevStudio.xml
  echo '         <glob pattern="*.rdd"/>'                                                   >> pragmadev-PragmaDevStudio.xml
  echo '  </mime-type>'                                                                     >> pragmadev-PragmaDevStudio.xml
  echo '  <mime-type type="application/pragmastudioprotoguipkg">'                           >> pragmadev-PragmaDevStudio.xml
  echo '         <comment xml:lang="en">PragmaDev Studio prototyping GUI package</comment>' >> pragmadev-PragmaDevStudio.xml
  echo '         <glob pattern="*.rdy"/>'                                                   >> pragmadev-PragmaDevStudio.xml
  echo '  </mime-type>'                                                                     >> pragmadev-PragmaDevStudio.xml
  echo '</mime-info>'                                                                       >> pragmadev-PragmaDevStudio.xml
  ## Install everything
  commands="xdg-desktop-menu install --mode $desktop_install PragmaDevStudio.desktop"
  commands="$commands;xdg-mime install --mode $desktop_install pragmadev-PragmaDevStudio.xml"
  commands="$commands;xdg-icon-resource install --mode $desktop_install --context mimetypes --size 24  $RTDS_HOME/share/icons/common/PragmaDevStudio-prj24.png  application-pragmastudioproject"
  commands="$commands;xdg-icon-resource install --mode $desktop_install --context mimetypes --size 48  $RTDS_HOME/share/icons/common/PragmaDevStudio-prj48.png  application-pragmastudioproject"
  commands="$commands;xdg-icon-resource install --mode $desktop_install --context mimetypes --size 128 $RTDS_HOME/share/icons/common/PragmaDevStudio-prj128.png application-pragmastudioproject"
  commands="$commands;xdg-icon-resource install --mode $desktop_install --context mimetypes --size 24  $RTDS_HOME/share/icons/common/PragmaDevStudio-dgm24.png  application-pragmastudiodiagram"
  commands="$commands;xdg-icon-resource install --mode $desktop_install --context mimetypes --size 48  $RTDS_HOME/share/icons/common/PragmaDevStudio-dgm48.png  application-pragmastudiodiagram"
  commands="$commands;xdg-icon-resource install --mode $desktop_install --context mimetypes --size 128 $RTDS_HOME/share/icons/common/PragmaDevStudio-dgm128.png application-pragmastudiodiagram"
  commands="$commands;xdg-icon-resource install --mode $desktop_install --context mimetypes --size 24  $RTDS_HOME/share/icons/common/PragmaDevStudio-dgm24.png  application/pragmastudioprotoguipkg"
  commands="$commands;xdg-icon-resource install --mode $desktop_install --context mimetypes --size 48  $RTDS_HOME/share/icons/common/PragmaDevStudio-dgm48.png  application/pragmastudioprotoguipkg"
  commands="$commands;xdg-icon-resource install --mode $desktop_install --context mimetypes --size 128 $RTDS_HOME/share/icons/common/PragmaDevStudio-dgm128.png application/pragmastudioprotoguipkg"
  if [ "$needs_sudo" = "YES" ]
  then
    echo "Enter your password for system-wide menu and file types installation:"
    sudo sh -c "$commands"
  else
    sh -c "$commands"
  fi
  ## Clean-up
  rm -f pragmadev-PragmaDevStudio.desktop pragmadev-PragmaDevStudio.xml
  popd
fi


## Done; display post-installation readme file
cat - $rootDir/doc/INSTALL.txt <<. | more
--- Installation complete.

.
echo 'Press return'
read dummy

## Over
echo ''
echo 'Installation completed successfully!'
echo ''
echo "WARNING: to be able to run $application_label, each account must have defined the"
echo "environment variable RTDS_HOME to $installDir, and included the path to"
echo "$application_label binaries in its PATH and LD_LIBRARY_PATH environment variables"
echo ""
echo "To do this:"
echo ""
echo " - on sh compatible shells (sh, ksh, bash), source in your .profile the script"
echo "   $installDir/pragmastudio-config.sh"
echo ""
echo " - on csh compatible shells (csh, tcsh), source in your .cshrc the script"
echo "   $installDir/pragmastudio-config.csh"
echo ''
echo 'Once your environment is correct, you can the following command(s):'
if [ "$install_studio" = "yes" ]
then
  echo " - 'pragmastudio' for PragmaDev Studio"
fi
if [ "$install_specifier" = "yes" ]
then
  echo " - 'pragmaspecifier' for PragmaDev Specifier"
fi
if [ "$install_developer" = "yes" ]
then
  echo " - 'pragmadeveloper' for PragmaDev Developer"
fi
if [ "$install_tester" = "yes" ]
then
  echo " - 'pragmatester' for PragmaDev Tester"
fi
echo ''
echo "Welcome to $application_label!"

exit 0
