#include "portable_sockets.h"

#include "constants.h"

#include "RTDS_MACRO.h"
#include "RTDS_gen.h"
#include "RTDS_messages.h"
#include "server_communication.h"

SOCKET_GLOBALS

/*
** VARIABLE socket_descriptor:
** ---------------------------
** Identifier for the socket connected to the distant server.
*/ 
static SOCKET socket_descriptor = INVALID_SOCKET;


/*
** FUNTION init_socket:
** --------------------
** Initializes the socket connection if needed.
** Returns: The identifier for the socket connected to the distant server.
*/

int init_socket(void)
  {
  struct addrinfo   hints;          /* Partial information on the distant server */
  struct addrinfo * result = NULL;  /* Connection information on the found distant server */
  int               cr;             /* Return code for the getaddrinfo call */
  
  /* If not connected yet */
  if ( socket_descriptor == INVALID_SOCKET )
    {
    SOCKET_STARTUP
    
    /* Search for the distant server */
    hints.ai_family = AF_INET;
    hints.ai_socktype = SOCK_STREAM;
    hints.ai_protocol = 0;
    hints.ai_flags = 0;
    hints.ai_addrlen = 0;
    hints.ai_addr = NULL;
    hints.ai_canonname = NULL;
    hints.ai_next = NULL;
    if ( ( cr = getaddrinfo("127.0.0.1", PORT_NUMBER_STRING, &hints, &result) ) != 0 )
      {
      /* If not found, error */
      fprintf(stderr, "init_socket - getaddrinfo: %d\n", cr);
      abort();
      }
    /* If distant server was found */
    else
      {
      /* Try to connect with every possible addressing */
      for ( ; result != NULL; result = result->ai_next )
        {
        /* Create the socket */
        socket_descriptor = socket(result->ai_family, result->ai_socktype, result->ai_protocol);
        if ( socket_descriptor == INVALID_SOCKET )
          {
          fprintf(stderr, "init_socket - socket: error %d\n", ERRNO);
          continue;
          }
        /* Connect it to the distant server */
        if ( connect(socket_descriptor, (struct sockaddr*)(result->ai_addr), result->ai_addrlen) == SOCKET_ERROR )
          {
          fprintf(stderr, "init_socket - connect: error %d\n", ERRNO);
          socket_descriptor = INVALID_SOCKET;
          continue;
          }
        }
      }
    }
  /* If socket could not be connected, error */
  if ( socket_descriptor == INVALID_SOCKET )
    {
    fprintf(stderr, "init_socket: no suitable addrinfo found!\n");
    abort();
    }
  /* Return identifier for socket */
  return socket_descriptor;
  }
  
  
/*
** FUNCTION send_to_server:
** ------------------------
** Actually sends a message to the distant server
** Parameters:
**  - message_number: Generated identifier for the message to send, as defined in RTDS_gen.h
**  - data_length: Length of the buffer for message parameters
**  - data_pointer: Pointer on the buffer for message parameters
*/

void send_to_server(int message_number, unsigned data_length, void * data_pointer)
  {
  RTDS_m_add_data * add_params;           /* Transport structure for m_add message */
  RTDS_m_mul_data * mul_params;           /* Transport structure for m_mul message */
  char              encoded_message[64];  /* Encoded message, sent through the socket to the distant server */
  int               sent_bytes;           /* Actual number of bytes sent, for error checking */
  
  /* Make sure socket is connected to the server */
  init_socket();
  /* Figure out which message is sent */
  switch(message_number)
    {
    /* Encoding of message and its parameters for m_add */
    case RTDS_message_m_add:
      add_params = (RTDS_m_add_data*)data_pointer;
      sprintf(encoded_message, "a%d:%d", add_params->param1, add_params->param2);
      break;
    /* Encoding of message and its parameters for m_mul */
    case RTDS_message_m_mul:
      mul_params = (RTDS_m_mul_data*)data_pointer;
      sprintf(encoded_message, "m%d:%d", mul_params->param1, mul_params->param2);
      break;
    /* Don't know any other message */
    default:
      return;
    }
  /* Send encoded message to the distant server through the socket */
  sent_bytes = send(socket_descriptor, encoded_message, 1 + strlen(encoded_message), 0);
  if ( sent_bytes == -1 )
    fprintf(stderr, "send_to_server - send: error %d\n", ERRNO);
  }
