#!/bin/sh

# =========================================
# INSTALLATION SCRIPT FOR PRAGMADEV PROCESS
# =========================================
# Copyright (c) PragmaDev 2019

## Function reading a line from the terminal
getAnswer()
{
  ## Read from terminal
  read answer
  ## Quit if needed
  if [ "$answer" = "q" ]
  then
    exit 0
  fi
  ## Output answer
  echo $answer
}

## Welcome screen
cat <<.
==============================
PRAGMADEV PROCESS INSTALLATION
==============================
This script will install PragmaDev Process.

Press return to continue or 'q' to quit.
.
getAnswer > /dev/null

## Architecture may be passed as an argument
arch=$1

## If it wasn't, try to figure it out
if [ "$arch" = "" ]
then
  case `uname -s` in

    Linux)
      arch='linux64'
      ;;

    SunOS)
      arch='solaris'
      ;;
  esac
fi

## If architecture still not found or invalid, can't proceed
if [ "$arch" != "linux64" -a "$arch" != "solaris" ]
then
  echo '!!! ERROR !!!'
  echo 'Unable to determine architecture.'
  echo 'You may try to re-run this installation script passing the architecture'
  echo 'as an argument. Valid architectures are "linux" and "solaris".'
  exit 1
fi

## Directory for install script is the mount point of the CD-ROM
rootDir=`dirname $0`

## Display license agreement
more $rootDir/doc/LICENSE.txt
echo ""
echo "To accept this license agreement, press return, or 'q' to quit the installation."
getAnswer > /dev/null

## Ask for installation directory
installDir=""
while [ "$installDir" = "" ]
do
  echo ""
  echo "Please enter the installation directory for PragmaDev Process"
  echo "(default: /opt/pragmadev/pragmaprocess; 'q' to quit)"
  installDir=`getAnswer`
  if [ "$installDir" = "" ]
  then
    installDir='/opt/pragmadev/pragmaprocess'
  fi

  ## If installation directory already exists
  if [ -r "$installDir" ]
  then
    ## If it's not a directory, error...
    if [ ! -d "$installDir" ]
    then
      echo '!!! ERROR !!!'
      echo "File $installDir exists, but is not a directory. Please choose another"
      echo "location."
      echo ""
      echo "Press return to continue or 'q' to quit"
      getAnswer > /dev/null
      installDir=""
    else
      ## If directory is not writable, error
      if [ ! -w "$installDir" ]
      then
        echo '!!! ERROR !!!'
        echo "Directory $installDir exists, but is not writable. Please choose another"
        echo "location, or quit the installation and re-log as a user with sufficient"
        echo "rights to write in this directory"
        echo ""
        echo "Press return to continue or 'q' to quit"
        getAnswer > /dev/null
        installDir=""
      fi
    fi
  ## If installation directory does not exist, ask if we should create it
  else
    echo "Directory $installDir does not exist. Do you want to create it ([y]/n/q)?"
    answer=`getAnswer`
    if [ "$answer" != "n" ]
    then
      mkdir -p $installDir
      ## If creation failed
      if [ $? -ne 0 ]
      then
        echo '!!! ERROR !!!'
        echo "Could not create installation directory $installDir"
        echo "Make sure you have the appropriate rights or choose another directory"
        echo ""
        echo "Press return to continue or 'q' to quit"
        getAnswer > /dev/null
        installDir=""
      fi
    else
      installDir=""
    fi
  fi
done

## Desktop menu, mimetypes & icon configuration
desktop_install=NO
if [ "`which xdg-desktop-menu 2> /dev/null`" = "" -o "`which xdg-mime 2> /dev/null`" = "" -o  "`which xdg-icon-resource 2> /dev/null`" = "" ]
then
  desktop_install=NO_XDG
else
  echo "The XDG utilities are installed on your system, so an entry to the desktop menu"
  echo "as well as the definitions for the file types for PragmaDev Process can be"
  echo "installed."
  if [ "`whoami`" = "root" ]
  then
    echo "Since you are root, the menu and file types will be installed system-wide."
    echo "Install menu entry & file types ([y]/n/q)?"
  else
    echo "Do you want to create a entry in the desktop menu and define the file types?"
    echo "  - n = no"
    echo "  - u = for this user only"
    echo "  - s = system-wide (requires root permissions)"
    echo "(n/[u]/s/q):"
  fi
  answer=`getAnswer`
  case $answer in
    u)
      desktop_install=user
      ;;
    y)
      desktop_install=system
      ;;
    s)
      desktop_install=sudo
      ;;
  esac
fi

## Confirmation screen
cat <<.

PragmaDev Process will now be installed with the following parameters:

 - Architecture: $arch
 
 - Installation directory: $installDir

PLEASE CHECK THE INFORMATIONS ABOVE. If you press return now, PragmaDev Process
will be installed. If any of the informations above is not correct, press 'q'
and re-run the installation script from the beginning.

.
getAnswer > /dev/null

## Install binaries
echo '--- Installing PragmaDev Process binaries'
cd $rootDir
if [ ! -r $installDir/bin ]
then
  mkdir $installDir/bin
fi
cp -r bin/$arch/* $installDir/bin/

## Install documentation
echo '--- Installing PragmaDev Process documentation'
tar cf - doc | (cd $installDir ; tar xf -)

## Install common files
echo '--- Installing PragmaDev Process common files'
tar cf - share | (cd $installDir ; tar xf -)

## Install examples
echo "--- Installing PragmaDev Process examples"
tar cf - examples | (cd $installDir ; tar xf -)

## Creating configuration scripts
echo '--- Creating configuration scripts'
echo "PRAGMADEV_PROCESS_HOME=$installDir"       >  $installDir/pragmaprocess-config.sh
echo 'PATH="$PATH":$PRAGMADEV_PROCESS_HOME/bin' >> $installDir/pragmaprocess-config.sh
echo 'export PRAGMADEV_PROCESS_HOME PATH'       >> $installDir/pragmaprocess-config.sh

echo "setenv PRAGMADEV_PROCESS_HOME $installDir"        >  $installDir/pragmaprocess-config.csh
echo 'setenv PATH "$PATH":$PRAGMADEV_PROCESS_HOME/bin'  >> $installDir/pragmaprocess-config.csh


## Desktop menu & icons
if [ "$desktop_install" = "user" -o "$desktop_install" = "system" -o "$desktop_install" = "sudo" ]
then
  echo '--- Installing desktop menu & icons'
  pushd /var/tmp
  needs_sudo=NO
  if [ "$desktop_install" = "sudo" ]
  then
    needs_sudo=YES
    desktop_install=system
  fi
  ## Desktop application configuration file
  echo "[Desktop Entry]"                                                              >  pragmadev-PragmaDevProcess.desktop
  echo "Type=Application"                                                             >> pragmadev-PragmaDevProcess.desktop
  echo "Name=PragmaDev Process"                                                       >> pragmadev-PragmaDevProcess.desktop
  echo "GenericName=Business process modelling tool based on BPMN"                    >> pragmadev-PragmaDevProcess.desktop
  echo "Icon=$PRAGMADEV_PROCESS_HOME/share/icons/common/PragmaDevProcess.svg"         >> pragmadev-PragmaDevProcess.desktop
  echo "Exec=$PRAGMADEV_PROCESS_HOME/bin/pragmaprocess "'"%f"'                        >> pragmadev-PragmaDevProcess.desktop
  echo "Terminal=false"                                                               >> pragmadev-PragmaDevProcess.desktop
  echo "MimeType=application/pragmaprocessproject;application/pragmaprocessmodel;"    >> pragmadev-PragmaDevProcess.desktop
  echo "Categories=Development;IDE"                                                   >> pragmadev-PragmaDevProcess.desktop
  ## Mimetypes file
  echo '<?xml version="1.0" encoding="UTF-8"?>'                                             >  pragmadev-PragmaDevProcess.xml
  echo '<mime-info xmlns="http://www.freedesktop.org/standards/shared-mime-info">'          >> pragmadev-PragmaDevProcess.xml
  echo '  <mime-type type="application/pragmaprocessproject">'                              >> pragmadev-PragmaDevProcess.xml
  echo '         <comment xml:lang="en">PragmaDev Process project</comment>'                >> pragmadev-PragmaDevProcess.xml
  echo '         <glob pattern="*.bprj"/>'                                                  >> pragmadev-PragmaDevProcess.xml
  echo '         <glob pattern="*.bprz"/>'                                                  >> pragmadev-PragmaDevProcess.xml
  echo '  </mime-type>'                                                                     >> pragmadev-PragmaDevProcess.xml
  echo '  <mime-type type="application/pragmaprocessmodel">'                                >> pragmadev-PragmaDevProcess.xml
  echo '         <comment xml:lang="en">PragmaDev Process model</comment>'                  >> pragmadev-PragmaDevProcess.xml
  echo '         <glob pattern="*.bpmn"/>'                                                  >> pragmadev-PragmaDevProcess.xml
  echo '  </mime-type>'                                                                     >> pragmadev-PragmaDevProcess.xml
  echo '</mime-info>'                                                                       >> pragmadev-PragmaDevProcess.xml
  ## Install everything
  commands="xdg-desktop-menu install --mode $desktop_install PragmaDevProcess.desktop"
  commands="$commands;xdg-mime install --mode $desktop_install pragmadev-PragmaDevProcess.xml"
  commands="$commands;xdg-icon-resource install --mode $desktop_install --context mimetypes --size 24  $PRAGMADEV_PROCESS_HOME/share/icons/common/Process-project-24.png  application-pragmaprocessproject"
  commands="$commands;xdg-icon-resource install --mode $desktop_install --context mimetypes --size 48  $PRAGMADEV_PROCESS_HOME/share/icons/common/Process-project-48.png  application-pragmaprocessproject"
  commands="$commands;xdg-icon-resource install --mode $desktop_install --context mimetypes --size 128 $PRAGMADEV_PROCESS_HOME/share/icons/common/Process-project-128.png application-pragmaprocessproject"
  commands="$commands;xdg-icon-resource install --mode $desktop_install --context mimetypes --size 24  $PRAGMADEV_PROCESS_HOME/share/icons/common/Process-model-24.png  application-pragmaprocessmodel"
  commands="$commands;xdg-icon-resource install --mode $desktop_install --context mimetypes --size 48  $PRAGMADEV_PROCESS_HOME/share/icons/common/Process-model-48.png  application-pragmaprocessmodel"
  commands="$commands;xdg-icon-resource install --mode $desktop_install --context mimetypes --size 128 $PRAGMADEV_PROCESS_HOME/share/icons/common/Process-model-128.png application-pragmaprocessmodel"
  if [ "$needs_sudo" = "YES" ]
  then
    echo "Enter your password for system-wide menu and file types installation:"
    sudo sh -c "$commands"
  else
    sh -c "$commands"
  fi
  ## Clean-up
  rm -f pragmadev-PragmaDevProcess.desktop pragmadev-PragmaDevProcess.xml
  popd
fi


## Done; display post-installation readme file
cat - $rootDir/doc/INSTALL.txt <<. | more
--- Installation complete.

.
echo 'Press return'
read dummy

## Over
echo ''
echo 'Installation completed successfully!'
echo ''
echo 'WARNING: to be able to run PragmaDev Process, each account must have'
echo 'defined the proper environment variables.'
echo ''
echo "To do this:"
echo ""
echo " - on sh compatible shells (sh, ksh, bash), source in your .profile the script"
echo "   $installDir/pragmaprocess-config.sh"
echo ""
echo " - on csh compatible shells (csh, tcsh), source in your .cshrc the script"
echo "   $installDir/pragmaprocess-config.csh"
echo ""
echo 'Welcome to PragmaDev Process!'

exit 0
